import axios from 'axios';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:5000/api/v1';

// Create axios instance with credentials
const api = axios.create({
  baseURL: API_URL,
  withCredentials: true,
  headers: {
    'Content-Type': 'application/json',
    'Authorization': `Bearer ${localStorage.getItem('token')}`,
  },
});
import { AxiosResponse } from 'axios';

interface SalesReportParams {
  startDate?: string;
  endDate?: string;
  period?: 'daily' | 'weekly' | 'monthly' | 'yearly';
  limit?: number;
  page?: number;
}

interface SalesData {
  date: string;
  sales: number;
  revenue: number;
  orders: number;
  customers: number;
}

interface TopProduct {
  id: string;
  name: string;
  sales: number;
  revenue: number;
  image?: string;
}

interface SalesReportResponse {
  success: boolean;
  data: {
    salesData: SalesData[];
    topProducts: TopProduct[];
    summary: {
      totalSales: number;
      totalRevenue: number;
      totalOrders: number;
      totalCustomers: number;
      averageOrderValue: number;
      conversionRate: number;
    };
  };
  message?: string;
}

export const reportService = {
  /**
   * Fetch sales report data
   * @param params Query parameters for filtering the report
   */
  getSalesReport: async (params?: SalesReportParams): Promise<SalesReportResponse> => {
    try {
      const response: AxiosResponse<SalesReportResponse> = await api.get('/reports/sales', { params });
      return response.data;
    } catch (error) {
      console.error('Error fetching sales report:', error);
      throw error;
    }
  },

  /**
   * Export sales report
   * @param format Export format (csv, excel, pdf)
   * @param params Query parameters for filtering the report
   */
  exportSalesReport: async (format: 'csv' | 'excel' | 'pdf' = 'excel', params?: SalesReportParams) => {
    try {
      const response = await api.get(`/reports/sales/export?format=${format}`, {
        params,
        responseType: 'blob',
      });
      
      // Create a download link and trigger download
      const url = window.URL.createObjectURL(new Blob([response.data]));
      const link = document.createElement('a');
      const fileName = `sales-report-${new Date().toISOString().split('T')[0]}.${format}`;
      
      link.href = url;
      link.setAttribute('download', fileName);
      document.body.appendChild(link);
      link.click();
      
      // Cleanup
      if (link.parentNode) {
        link.parentNode.removeChild(link);
      }
      window.URL.revokeObjectURL(url);
      
      return { success: true };
    } catch (error) {
      console.error('Error exporting sales report:', error);
      throw error;
    }
  },

  /**
   * Get available report periods
   */
  getReportPeriods: async () => {
    try {
      const response = await api.get('/reports/periods');
      return response.data;
    } catch (error) {
      console.error('Error fetching report periods:', error);
      return [
        { value: 'today', label: 'Today' },
        { value: 'yesterday', label: 'Yesterday' },
        { value: 'this_week', label: 'This Week' },
        { value: 'last_week', label: 'Last Week' },
        { value: 'this_month', label: 'This Month' },
        { value: 'last_month', label: 'Last Month' },
        { value: 'this_year', label: 'This Year' },
        { value: 'last_year', label: 'Last Year' },
        { value: 'custom', label: 'Custom Range' },
      ];
    }
  },
};
