import api from './api';
import { Banner, BannerCreateDTO, BannerListResponse, BannerResponse } from '@/types/banner';

// Get active banners by type
export const getActiveBanners = async (type: 'main' | 'secondary' | 'promo' = 'main'): Promise<Banner[]> => {
  try {
    const response = await api.get<BannerListResponse>('/banners/active', {
      params: { type }
    });
    return response.data.data;
  } catch (error) {
    console.error('Error fetching active banners:', error);
    return [];
  }
};

// Get all banners (admin only)
export const getAllBanners = async () => {
  try {
    const response = await api.get<BannerListResponse>('/banners');
    return response.data;
  } catch (error) {
    console.error('Error fetching all banners:', error);
    throw error;
  }
};

// Get banner by ID
export const getBannerById = async (id: string) => {
  try {
    const response = await api.get<BannerResponse>(`/banners/${id}`);
    return response.data;
  } catch (error) {
    console.error('Error fetching banner by ID:', error);
    throw error;
  }
};

// Create new banner (admin only)
export const createBanner = async (bannerData: BannerCreateDTO) => {
  try {
    const response = await api.post<BannerResponse>('/banners',
      bannerData
    );
    return response.data;
  } catch (error) {
    console.error('Error creating banner:', error);
    throw error;
  }
};

// Update banner (admin only)
export const updateBanner = async (id: string, bannerData: Partial<BannerCreateDTO>) => {
  try {
    const response = await api.patch<BannerResponse>(
      `/banners/${id}`,
      bannerData
    );
    return response.data;
  } catch (error) {
    console.error('Error updating banner:', error);
    throw error;
  }
};

// Delete banner (admin only)
export const deleteBanner = async (id: string) => {
  try {
    const response = await api.delete(`/banners/${id}`);
    return response.data;
  } catch (error) {
    console.error('Error deleting banner:', error);
    throw error;
  }
};

// Upload image for a banner and return the updated banner with image URL
export const uploadBannerImage = async (bannerId: string, file: File): Promise<Banner> => {
  const formData = new FormData();
  formData.append('image', file);

  try {
    const response = await api.patch<BannerResponse>(
      `/banners/${bannerId}/photo`,
      formData,
      {
        headers: {
          'Content-Type': 'multipart/form-data'
        }
      }
    );
    return response.data.data;
  } catch (error) {
    console.error('Error uploading banner image:', error);
    throw error;
  }
};
