import React, { useState, useEffect } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { storeSidebarItems } from '../StoreLayout';
import { Search, Plus, Filter, Download, Edit, Eye, EyeOff, Trash2, User, Mail, Phone, Package, Clock } from 'lucide-react';
import { toast } from 'react-hot-toast';
import api from '@/services/api';
import { useNavigate } from 'react-router-dom';

interface PickerPacker {
  _id: string;
  name: string;
  email: string;
  phone: string;
  employee_id: string;
  shift: string;
  performance_rating: number;
  status: string;
  image: string;
  joining_date: string;
  orders_processed: number;
  store: {
    _id: string;
    storeName: string;
  };
}

const PickerPackerPage: React.FC = () => {
  const [pickerPackers, setPickerPackers] = useState<PickerPacker[]>([]); // Initialize as empty array
  const [loading, setLoading] = useState<boolean>(true);
  const [searchTerm, setSearchTerm] = useState<string>('');
  const [statusFilter, setStatusFilter] = useState<string>('all');
  const [shiftFilter, setShiftFilter] = useState<string>('all');
  const [currentPage, setCurrentPage] = useState<number>(1);
  const [totalPages, setTotalPages] = useState<number>(1);
  const [selectedPickerPacker, setSelectedPickerPacker] = useState<PickerPacker | null>(null);
  const [isViewModalOpen, setIsViewModalOpen] = useState<boolean>(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState<boolean>(false);
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState<boolean>(false);
  const [isCreateModalOpen, setIsCreateModalOpen] = useState<boolean>(false);
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    password: '',
    phone: '',
    employee_id: '',
    shift: 'morning',
    performance_rating: 0,
    status: 'active',
    image: null as File | null,
    imagePreview: '' as string | null,
    joining_date: '',
    orders_processed: 0
  });
  
  const [showPassword, setShowPassword] = useState(false);
  const [isUploading, setIsUploading] = useState(false);
  const navigate = useNavigate();

  // Fetch picker packers
  const fetchPickerPackers = async (page = 1, search = '', status = 'all', shift = 'all') => {
    setLoading(true);
    try {
      // First, get the current user's store ID if they are a store manager/admin
      let storeId = '';
      try {
        const userResponse = await api.get('/auth/me');
        const userData = userResponse.data?.data || {};
        
        // Get store ID based on user role
        if (userData.store?._id) {
          storeId = userData.store._id;
        } else if (userData.storeManager) {
          storeId = userData.storeManager;
        } else if (userData.role === 'admin') {
          storeId = ''; // Empty store ID for admin to see all
        }
      } catch (userError) {
        console.error('Error fetching user info:', userError);
      }

      // Prepare query params
      const params: any = {
        page,
        limit: 10,
        search,
        status: status !== 'all' ? status : undefined,
        shift: shift !== 'all' ? shift : undefined,
        populate: 'store' // Populate store data if needed
      };

      // Only add store filter if we have a store ID
      if (storeId) {
        params.store = storeId;
      }

      const response = await api.get('/picker-packers', { params });
      console.log('PickerPackers API response:', response.data);
      
      // Handle response
      let pickerPackersData: PickerPacker[] = [];
      let totalPages = 1;
      
      // Process the response data
      if (response.data?.success) {
        const responseData = response.data.data || response.data;
        
        // Handle different response structures
        if (Array.isArray(responseData)) {
          pickerPackersData = responseData;
        } else if (responseData?.items) {
          pickerPackersData = responseData.items;
        } else if (responseData?.pickerPackers) {
          pickerPackersData = responseData.pickerPackers;
        } else if (responseData?.data) {
          pickerPackersData = Array.isArray(responseData.data) ? responseData.data : [responseData.data];
        }
        
        // Ensure we have an array
        pickerPackersData = Array.isArray(pickerPackersData) ? pickerPackersData : [pickerPackersData];
        
        // Filter out any null/undefined values
        pickerPackersData = pickerPackersData.filter(Boolean);
        
        // Get total pages from response
        totalPages = responseData.totalPages || 
                    response.data.totalPages || 
                    (responseData.pagination?.pages || 
                      (responseData.total ? Math.ceil(responseData.total / (params.limit || 10)) : 1) || 
                      1);
      }
      
      setPickerPackers(pickerPackersData);
      setTotalPages(totalPages);
      setCurrentPage(page);
    } catch (error) {
      console.error('Error fetching picker packers:', error);
      toast.error('Failed to load picker packers');
      setPickerPackers([]);
      setTotalPages(1);
      setCurrentPage(1);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchPickerPackers();
  }, []);

  // Handle search
  const handleSearch = (value: string) => {
    setSearchTerm(value);
    fetchPickerPackers(1, value, statusFilter, shiftFilter);
  };

  // Handle status filter change
  const handleStatusFilterChange = (value: string) => {
    setStatusFilter(value);
    fetchPickerPackers(1, searchTerm, value, shiftFilter);
  };

  // Handle shift filter change
  const handleShiftFilterChange = (value: string) => {
    setShiftFilter(value);
    fetchPickerPackers(1, searchTerm, statusFilter, value);
  };

  // Handle page change
  const handlePageChange = (page: number) => {
    fetchPickerPackers(page, searchTerm, statusFilter, shiftFilter);
  };

  // Handle view picker packer
  const handleViewPickerPacker = (pickerPacker: PickerPacker) => {
    setSelectedPickerPacker(pickerPacker);
    setIsViewModalOpen(true);
  };

  // Handle edit picker packer
  const handleEditPickerPacker = (pickerPacker: PickerPacker) => {
    setSelectedPickerPacker(pickerPacker);
    setFormData({
      name: pickerPacker.name,
      email: pickerPacker.email,
      password: '', // Don't pre-fill password for security reasons
      phone: pickerPacker.phone || '',
      employee_id: pickerPacker.employee_id || '',
      shift: pickerPacker.shift || 'morning',
      performance_rating: pickerPacker.performance_rating || 0,
      status: pickerPacker.status,
      image: null,
      imagePreview: pickerPacker.image || null,
      joining_date: pickerPacker.joining_date ? new Date(pickerPacker.joining_date).toISOString().split('T')[0] : '',
      orders_processed: pickerPacker.orders_processed || 0
    });
    setIsEditModalOpen(true);
  };

  // Handle delete picker packer
  const handleDeletePickerPacker = (pickerPacker: PickerPacker) => {
    setSelectedPickerPacker(pickerPacker);
    setIsDeleteModalOpen(true);
  };

  // Handle create picker packer
  const handleCreatePickerPacker = () => {
    setFormData({
      name: '',
      email: '',
      password: '',
      phone: '',
      employee_id: '',
      shift: 'morning',
      performance_rating: 0,
      status: 'active',
      image: null,
      imagePreview: null,
      joining_date: '',
      orders_processed: 0
    });
    setShowPassword(false);
    setIsCreateModalOpen(true);
  };

  // Handle file input change
  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      
      // Check file type
      if (!file.type.match('image.*')) {
        toast.error('Please select a valid image file');
        return;
      }
      
      // Check file size (max 5MB)
      if (file.size > 5 * 1024 * 1024) {
        toast.error('Image size should be less than 5MB');
        return;
      }
      
      setFormData(prev => ({
        ...prev,
        image: file,
        imagePreview: URL.createObjectURL(file)
      }));
    }
  };

  // Handle form input change
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    const { name, value, type } = e.target as HTMLInputElement;
    
    if (type === 'file') {
      return; // Handled by handleFileChange
    }
    
    setFormData(prev => ({
      ...prev,
      [name]: ['performance_rating', 'orders_processed'].includes(name) ? parseInt(value) || 0 : value
    }));
  };

  // Handle form submission for create/edit
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    try {
      setIsUploading(true);
      const formDataToSend = new FormData();
      
      // Append all form data to FormData object
      Object.entries(formData).forEach(([key, value]) => {
        if (value !== null && value !== undefined) {
          if (key === 'image' && value instanceof File) {
            formDataToSend.append('image', value);
          } else if (key !== 'imagePreview') {
            formDataToSend.append(key, String(value));
          }
        }
      });
      
      if (selectedPickerPacker) {
        // Edit picker packer
        await api.put(`/picker-packers/${selectedPickerPacker._id}`, formDataToSend, {
          headers: {
            'Content-Type': 'multipart/form-data',
          },
        });
        toast.success('Picker/Packer updated successfully');
        setIsEditModalOpen(false);
      } else {
        // Create picker packer
        await api.post('/picker-packers', formDataToSend, {
          headers: {
            'Content-Type': 'multipart/form-data',
          },
        });
        toast.success('Picker/Packer created successfully');
        setIsCreateModalOpen(false);
      }
      
      fetchPickerPackers(currentPage, searchTerm, statusFilter, shiftFilter);
    } catch (error) {
      console.error('Error saving picker packer:', error);
      toast.error('Failed to save picker packer');
    } finally {
      setIsUploading(false);
    }
  };

  // Handle delete confirmation
  const confirmDelete = async () => {
    if (!selectedPickerPacker) return;
    
    try {
      await api.delete(`/picker-packers/${selectedPickerPacker._id}`);
      toast.success('Picker/Packer deleted successfully');
      setIsDeleteModalOpen(false);
      fetchPickerPackers(currentPage, searchTerm, statusFilter, shiftFilter);
    } catch (error) {
      console.error('Error deleting picker packer:', error);
      toast.error('Failed to delete picker packer');
    }
  };

  // Get performance rating color
  const getPerformanceRatingColor = (rating: number) => {
    if (rating >= 4) return 'bg-green-100 text-green-800';
    if (rating >= 3) return 'bg-blue-100 text-blue-800';
    if (rating >= 2) return 'bg-yellow-100 text-yellow-800';
    return 'bg-red-100 text-red-800';
  };

  // Get shift display name
  const getShiftDisplayName = (shift: string) => {
    switch (shift) {
      case 'morning': return 'Morning';
      case 'afternoon': return 'Afternoon';
      case 'night': return 'Night';
      default: return shift;
    }
  };

  return (
    <Layout title="Picker/Packer Management" sidebar={<Sidebar items={storeSidebarItems} />}>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <h1 className="text-2xl font-bold">Picker/Packer Management</h1>
          <button
            onClick={handleCreatePickerPacker}
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            <Plus className="w-4 h-4 mr-2" />
            Add Picker/Packer
          </button>
        </div>

        {/* Filters */}
        <div className="flex flex-col sm:flex-row gap-4">
          <div className="relative flex-grow">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              type="text"
              placeholder="Search by name, email, employee ID..."
              className="pl-10 pr-4 py-2 w-full border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              value={searchTerm}
              onChange={(e) => handleSearch(e.target.value)}
            />
          </div>
          <select
            value={statusFilter}
            onChange={(e) => handleStatusFilterChange(e.target.value)}
            className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
          >
            <option value="all">All Status</option>
            <option value="active">Active</option>
            <option value="inactive">Inactive</option>
            <option value="on_leave">On Leave</option>
          </select>
          <select
            value={shiftFilter}
            onChange={(e) => handleShiftFilterChange(e.target.value)}
            className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
          >
            <option value="all">All Shifts</option>
            <option value="morning">Morning</option>
            <option value="afternoon">Afternoon</option>
            <option value="night">Night</option>
          </select>
          <button
            onClick={() => {
              setSearchTerm('');
              setStatusFilter('all');
              setShiftFilter('all');
              fetchPickerPackers(1, '', 'all', 'all');
            }}
            className="flex items-center px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
          >
            <Filter className="w-4 h-4 mr-2" />
            Reset Filters
          </button>
          <button
            onClick={() => toast.success('Export functionality coming soon')}
            className="flex items-center px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
          >
            <Download className="w-4 h-4 mr-2" />
            Export
          </button>
        </div>

        {/* Picker Packers Table */}
        <div className="bg-white shadow-sm border rounded-lg overflow-hidden">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Employee
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Contact
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Employee ID
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Shift
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Performance
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Orders Processed
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Status
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Actions
                  </th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {loading ? (
                  <tr>
                    <td colSpan={8} className="px-6 py-12 text-center text-gray-500">
                      <div className="flex justify-center">
                        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                      </div>
                      <p className="mt-2">Loading picker/packers...</p>
                    </td>
                  </tr>
                ) : !Array.isArray(pickerPackers) || pickerPackers.length === 0 ? (
                  <tr>
                    <td colSpan={8} className="px-6 py-12 text-center text-gray-500">
                      No picker/packers found
                    </td>
                  </tr>
                ) : (
                  pickerPackers.map((pickerPacker) => (
                    <tr key={pickerPacker._id}>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          <div className="h-10 w-10 flex-shrink-0">
                            {pickerPacker.image ? (
                              <img
                                className="h-10 w-10 rounded-full object-cover"
                                src={pickerPacker.image}
                                alt={pickerPacker.name}
                              />
                            ) : (
                              <div className="h-10 w-10 rounded-full bg-blue-100 flex items-center justify-center">
                                <User className="h-5 w-5 text-blue-500" />
                              </div>
                            )}
                          </div>
                          <div className="ml-4">
                            <div className="text-sm font-medium text-gray-900">{pickerPacker.name}</div>
                            <div className="text-sm text-gray-500">
                              {pickerPacker.joining_date ? `Joined: ${new Date(pickerPacker.joining_date).toLocaleDateString()}` : 'No joining date'}
                            </div>
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900 flex items-center">
                          <Mail className="h-4 w-4 mr-1 text-gray-500" />
                          {pickerPacker.email}
                        </div>
                        {pickerPacker.phone && (
                          <div className="text-sm text-gray-500 flex items-center mt-1">
                            <Phone className="h-4 w-4 mr-1 text-gray-500" />
                            {pickerPacker.phone}
                          </div>
                        )}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {pickerPacker.employee_id || 'N/A'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        <div className="flex items-center">
                          <Clock className="h-4 w-4 mr-1 text-gray-500" />
                          {getShiftDisplayName(pickerPacker.shift || 'morning')}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                          getPerformanceRatingColor(pickerPacker.performance_rating || 0)
                        }`}>
                          {pickerPacker.performance_rating ? `${pickerPacker.performance_rating}/5` : 'Not rated'}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        <div className="flex items-center">
                          <Package className="h-4 w-4 mr-1 text-gray-500" />
                          {pickerPacker.orders_processed || 0}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                          pickerPacker.status === 'active'
                            ? 'bg-green-100 text-green-800'
                            : pickerPacker.status === 'on_leave'
                            ? 'bg-yellow-100 text-yellow-800'
                            : 'bg-red-100 text-red-800'
                        }`}>
                          {pickerPacker.status === 'active' 
                            ? 'Active' 
                            : pickerPacker.status === 'on_leave'
                            ? 'On Leave'
                            : 'Inactive'}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                        <div className="flex space-x-2">
                          <button
                            onClick={() => handleViewPickerPacker(pickerPacker)}
                            className="text-blue-600 hover:text-blue-900"
                            title="View Picker/Packer"
                          >
                            <Eye className="w-4 h-4" />
                          </button>
                          <button
                            onClick={() => handleEditPickerPacker(pickerPacker)}
                            className="text-indigo-600 hover:text-indigo-900"
                            title="Edit Picker/Packer"
                          >
                            <Edit className="w-4 h-4" />
                          </button>
                          <button
                            onClick={() => handleDeletePickerPacker(pickerPacker)}
                            className="text-red-600 hover:text-red-900"
                            title="Delete Picker/Packer"
                          >
                            <Trash2 className="w-4 h-4" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
          
          {/* Pagination */}
          {totalPages > 1 && (
            <div className="bg-white px-4 py-3 border-t border-gray-200 sm:px-6">
              <div className="flex items-center justify-between">
                <div className="flex items-center">
                  <p className="text-sm text-gray-700">
                    Page <span className="font-medium">{currentPage}</span> of{' '}
                    <span className="font-medium">{totalPages}</span>
                  </p>
                </div>
                <div className="flex items-center space-x-2">
                  <button
                    onClick={() => handlePageChange(currentPage - 1)}
                    disabled={currentPage === 1}
                    className={`px-3 py-1 rounded-md ${
                      currentPage === 1
                        ? 'bg-gray-100 text-gray-400 cursor-not-allowed'
                        : 'bg-white text-gray-700 hover:bg-gray-50 border border-gray-300'
                    }`}
                  >
                    Previous
                  </button>
                  <button
                    onClick={() => handlePageChange(currentPage + 1)}
                    disabled={currentPage === totalPages}
                    className={`px-3 py-1 rounded-md ${
                      currentPage === totalPages
                        ? 'bg-gray-100 text-gray-400 cursor-not-allowed'
                        : 'bg-white text-gray-700 hover:bg-gray-50 border border-gray-300'
                    }`}
                  >
                    Next
                  </button>
                </div>
              </div>
            </div>
          )}
        </div>
      </div>

      {/* View Picker/Packer Modal */}
      {isViewModalOpen && selectedPickerPacker && (
        <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
          <div className="relative top-20 mx-auto p-5 border w-11/12 max-w-4xl shadow-lg rounded-md bg-white">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold text-gray-900">Picker/Packer Details</h3>
              <button
                onClick={() => setIsViewModalOpen(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                ×
              </button>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="space-y-4">
                <div className="flex justify-center">
                  {selectedPickerPacker.image ? (
                    <img
                      src={selectedPickerPacker.image}
                      alt={selectedPickerPacker.name}
                      className="h-48 w-48 object-cover rounded-lg"
                    />
                  ) : (
                    <div className="h-48 w-48 rounded-lg bg-blue-100 flex items-center justify-center">
                      <User className="h-16 w-16 text-blue-500" />
                    </div>
                  )}
                </div>
                
                <div>
                  <h4 className="font-medium text-gray-900 mb-2">Personal Information</h4>
                  <div className="space-y-2 text-sm">
                    <div><span className="font-medium">Name:</span> {selectedPickerPacker.name}</div>
                    <div><span className="font-medium">Email:</span> {selectedPickerPacker.email}</div>
                    <div><span className="font-medium">Phone:</span> {selectedPickerPacker.phone || 'N/A'}</div>
                    <div><span className="font-medium">Employee ID:</span> {selectedPickerPacker.employee_id || 'N/A'}</div>
                    <div><span className="font-medium">Joining Date:</span> {
                      selectedPickerPacker.joining_date 
                        ? new Date(selectedPickerPacker.joining_date).toLocaleDateString() 
                        : 'N/A'
                    }</div>
                  </div>
                </div>
              </div>
              
              <div className="space-y-4">
                <div>
                  <h4 className="font-medium text-gray-900 mb-2">Work Details</h4>
                  <div className="space-y-2 text-sm">
                    <div><span className="font-medium">Shift:</span> {getShiftDisplayName(selectedPickerPacker.shift || 'morning')}</div>
                    <div>
                      <span className="font-medium">Performance Rating:</span>{' '}
                      <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                        getPerformanceRatingColor(selectedPickerPacker.performance_rating || 0)
                      }`}>
                        {selectedPickerPacker.performance_rating ? `${selectedPickerPacker.performance_rating}/5` : 'Not rated'}
                      </span>
                    </div>
                    <div><span className="font-medium">Orders Processed:</span> {selectedPickerPacker.orders_processed || 0}</div>
                    <div>
                      <span className="font-medium">Status:</span>{' '}
                      <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                        selectedPickerPacker.status === 'active'
                          ? 'bg-green-100 text-green-800'
                          : selectedPickerPacker.status === 'on_leave'
                          ? 'bg-yellow-100 text-yellow-800'
                          : 'bg-red-100 text-red-800'
                      }`}>
                        {selectedPickerPacker.status === 'active' 
                          ? 'Active' 
                          : selectedPickerPacker.status === 'on_leave'
                          ? 'On Leave'
                          : 'Inactive'}
                      </span>
                    </div>
                  </div>
                </div>
                
                <div>
                  <h4 className="font-medium text-gray-900 mb-2">Store Information</h4>
                  <div className="space-y-2 text-sm">
                    <div><span className="font-medium">Store:</span> {selectedPickerPacker.store?.storeName || 'N/A'}</div>
                  </div>
                </div>
              </div>
            </div>
            
            <div className="mt-6 flex justify-end space-x-3">
              <button
                onClick={() => {
                  setIsViewModalOpen(false);
                  handleEditPickerPacker(selectedPickerPacker);
                }}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
              >
                Edit Picker/Packer
              </button>
              <button
                onClick={() => setIsViewModalOpen(false)}
                className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Create/Edit Picker/Packer Modal */}
      {(isCreateModalOpen || isEditModalOpen) && (
        <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
          <div className="relative top-20 mx-auto p-5 border w-11/12 max-w-4xl shadow-lg rounded-md bg-white">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold text-gray-900">
                {isEditModalOpen ? 'Edit Picker/Packer' : 'Add New Picker/Packer'}
              </h3>
              <button
                onClick={() => {
                  isEditModalOpen ? setIsEditModalOpen(false) : setIsCreateModalOpen(false);
                }}
                className="text-gray-400 hover:text-gray-600"
              >
                ×
              </button>
            </div>
            
            <form onSubmit={handleSubmit} className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Name *
                  </label>
                  <input
                    type="text"
                    name="name"
                    value={formData.name}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    required
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Email *
                  </label>
                  <input
                    type="email"
                    name="email"
                    value={formData.email}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    required
                  />
                </div>
                
                {!selectedPickerPacker && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Password *
                    </label>
                    <div className="relative">
                      <input
                        type={showPassword ? 'text' : 'password'}
                        name="password"
                        value={formData.password}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 pr-10"
                        required={!selectedPickerPacker}
                        minLength={6}
                      />
                      <button
                        type="button"
                        onClick={() => setShowPassword(!showPassword)}
                        className="absolute inset-y-0 right-0 pr-3 flex items-center text-gray-600 hover:text-gray-800"
                        tabIndex={-1}
                      >
                        {showPassword ? (
                          <EyeOff className="h-5 w-5" />
                        ) : (
                          <Eye className="h-5 w-5" />
                        )}
                      </button>
                    </div>
                    <p className="mt-1 text-xs text-gray-500">
                      {formData.password ? `${formData.password.length} characters` : 'Minimum 6 characters'}
                    </p>
                  </div>
                )}
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Phone
                  </label>
                  <input
                    type="tel"
                    name="phone"
                    value={formData.phone}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Employee ID
                  </label>
                  <input
                    type="text"
                    name="employee_id"
                    value={formData.employee_id}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Shift
                  </label>
                  <select
                    name="shift"
                    value={formData.shift}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  >
                    <option value="morning">Morning</option>
                    <option value="afternoon">Afternoon</option>
                    <option value="night">Night</option>
                  </select>
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Performance Rating (1-5)
                  </label>
                  <input
                    type="number"
                    name="performance_rating"
                    value={formData.performance_rating}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    min="0"
                    max="5"
                    step="0.1"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Status
                  </label>
                  <select
                    name="status"
                    value={formData.status}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  >
                    <option value="active">Active</option>
                    <option value="inactive">Inactive</option>
                    <option value="on_leave">On Leave</option>
                  </select>
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Orders Processed
                  </label>
                  <input
                    type="number"
                    name="orders_processed"
                    value={formData.orders_processed}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    min="0"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Profile Image
                  </label>
                  <div className="mt-1 flex items-center">
                    <div className="relative
                      w-20 h-20 rounded-full overflow-hidden
                      border-2 border-dashed border-gray-300
                      flex items-center justify-center
                      bg-gray-50 hover:bg-gray-100 transition-colors
                      cursor-pointer">
                      {formData.imagePreview ? (
                        <img 
                          src={formData.imagePreview} 
                          alt="Preview" 
                          className="w-full h-full object-cover"
                        />
                      ) : formData.image ? (
                        <img 
                          src={typeof formData.image === 'string' ? formData.image : URL.createObjectURL(formData.image)}
                          alt="Preview"
                          className="w-full h-full object-cover"
                        />
                      ) : (
                        <div className="flex flex-col items-center text-gray-400">
                          <Plus className="h-5 w-5" />
                          <span className="text-xs mt-1">Add Photo</span>
                        </div>
                      )}
                      <input
                        type="file"
                        name="image"
                        accept="image/*"
                        onChange={handleFileChange}
                        className="absolute inset-0 w-full h-full opacity-0 cursor-pointer"
                      />
                    </div>
                    {formData.image && (
                      <button
                        type="button"
                        onClick={(e) => {
                          e.stopPropagation();
                          setFormData(prev => ({
                            ...prev,
                            image: null,
                            imagePreview: null
                          }));
                        }}
                        className="ml-2 p-1 text-red-500 hover:text-red-700"
                        title="Remove image"
                      >
                        <Trash2 className="h-4 w-4" />
                      </button>
                    )}
                  </div>
                  <p className="mt-1 text-xs text-gray-500">
                    Click to upload an image (max 5MB)
                  </p>
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Joining Date
                  </label>
                  <input
                    type="date"
                    name="joining_date"
                    value={formData.joining_date}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  />
                </div>
              </div>
              
              <div className="flex justify-end space-x-3">
                <button
                  type="button"
                  onClick={() => {
                    isEditModalOpen ? setIsEditModalOpen(false) : setIsCreateModalOpen(false);
                  }}
                  className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  disabled={isUploading}
                  className={`px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 ${isUploading ? 'opacity-70 cursor-not-allowed' : ''}`}
                >
                  {isUploading ? (
                    <span className="flex items-center">
                      <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                      </svg>
                      {isEditModalOpen ? 'Updating...' : 'Creating...'}
                    </span>
                  ) : isEditModalOpen ? 'Update Picker/Packer' : 'Create Picker/Packer'}
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* Delete Confirmation Modal */}
      {isDeleteModalOpen && selectedPickerPacker && (
        <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
          <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div className="mt-3 text-center">
              <div className="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-red-100">
                <Trash2 className="h-6 w-6 text-red-600" />
              </div>
              <h3 className="text-lg leading-6 font-medium text-gray-900 mt-4">Delete Picker/Packer</h3>
              <div className="mt-2 px-7 py-3">
                <p className="text-sm text-gray-500">
                  Are you sure you want to delete {selectedPickerPacker.name}? This action cannot be undone.
                </p>
              </div>
              <div className="flex justify-center mt-4 space-x-3">
                <button
                  onClick={() => setIsDeleteModalOpen(false)}
                  className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button
                  onClick={confirmDelete}
                  className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700"
                >
                  Delete
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </Layout>
  );
};

export default PickerPackerPage;