import React, { useState, useEffect } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import { ArrowLeft, Upload, MapPin, Building, User, CreditCard, Settings } from 'lucide-react';
import { toast } from "react-hot-toast";
import api from "@/services/api";
import { useNavigate, useParams } from 'react-router-dom';

interface StoreFormData {
  // Seller Credentials
  name: string;
  email: string;
  mobile: string;
  password: string;
  
  // Store Information
  store: {
    storeName: string;
    categoryIds: string[];
    productStatus: 'Active' | 'Inactive';
    taxName: string;
    taxNumber: string;
    panNumber: string;
    commission: number;
    description: string;
  };
  
  // Store Location
  location: {
    city: string;
    state: string;
    street: string;
    latitude: number;
    longitude: number;
  };
  
  // Bank Information
  bank: {
    bankName: string;
    accountNumber: string;
    ifscCode: string;
    accountHolderName: string;
  };
  
  // Settings
  settings: {
    requireProductApproval: boolean;
    viewCustomerDetails: boolean;
  };
  
  status: 'Active' | 'Inactive' | 'Pending' | 'Suspended';
}

const EditStoreManagers: React.FC = () => {
  const navigate = useNavigate();
  const { id } = useParams<{ id: string }>();
  const [loading, setLoading] = useState(false);
  const [initialLoading, setInitialLoading] = useState(true);
  const [selectedFiles, setSelectedFiles] = useState<{
    logo?: File;
    nationalIdCard?: File;
    addressProof?: File;
  }>({});
  
  const [formData, setFormData] = useState<StoreFormData>({
    name: '',
    email: '',
    mobile: '',
    password: '',
    store: {
      storeName: '',
      categoryIds: [],
      productStatus: 'Active',
      taxName: '',
      taxNumber: '',
      panNumber: '',
      commission: 10,
      description: ''
    },
    location: {
      city: '',
      state: '',
      street: '',
      latitude: 0,
      longitude: 0
    },
    bank: {
      bankName: '',
      accountNumber: '',
      ifscCode: '',
      accountHolderName: ''
    },
    settings: {
      requireProductApproval: false,
      viewCustomerDetails: false
    },
    status: 'Pending'
  });

  // Fetch store data
  const fetchStore = async () => {
    try {
      setInitialLoading(true);
      const response = await api.get(`/stores/${id}`);
      const store = response.data.data;
      
      setFormData({
        name: store.name || '',
        email: store.email || '',
        mobile: store.mobile || '',
        password: '', // Don't populate password for security
        store: {
          storeName: store.store?.storeName || '',
          categoryIds: store.store?.categoryIds || [],
          productStatus: store.store?.productStatus || 'Active',
          taxName: store.store?.taxName || '',
          taxNumber: store.store?.taxNumber || '',
          panNumber: store.store?.panNumber || '',
          commission: store.store?.commission || 10,
          description: store.store?.description || ''
        },
        location: {
          city: store.location?.city || '',
          state: store.location?.state || '',
          street: store.location?.street || '',
          latitude: store.location?.latitude || 0,
          longitude: store.location?.longitude || 0
        },
        bank: {
          bankName: store.bank?.bankName || '',
          accountNumber: store.bank?.accountNumber || '',
          ifscCode: store.bank?.ifscCode || '',
          accountHolderName: store.bank?.accountHolderName || ''
        },
        settings: {
          requireProductApproval: store.settings?.requireProductApproval || false,
          viewCustomerDetails: store.settings?.viewCustomerDetails || false
        },
        status: store.status || 'Pending'
      });
    } catch (error: any) {
      console.error('Error fetching store:', error);
      toast.error('Failed to fetch store data');
      navigate('/admin/store-managers');
    } finally {
      setInitialLoading(false);
    }
  };

  useEffect(() => {
    if (id) {
      fetchStore();
    }
  }, [id]);

  // Handle input changes
  const handleInputChange = (section: keyof StoreFormData, field: string, value: any) => {
    // If we're updating a file field in the store object, handle it specially
    if (section === 'store' && ['logo', 'nationalIdCard', 'addressProof'].includes(field)) {
      // If the field is being cleared, mark it for removal
      if (!value) {
        setFilesToRemove(prev => ({
          ...prev,
          [field]: true
        }));
      } else {
        // If a new value is set, make sure we don't mark it for removal
        setFilesToRemove(prev => ({
          ...prev,
          [field]: false
        }));
      }
    }
    
    setFormData(prev => ({
      ...prev,
      [section]: {
        ...(prev[section] as any),
        [field]: value
      }
    }));
  };

  // Track which files should be removed
  const [filesToRemove, setFilesToRemove] = useState<{[key: string]: boolean}>({
    logo: false,
    nationalIdCard: false,
    addressProof: false
  });

  // Handle file uploads
  const handleFileChange = (fileType: string, file: File | null) => {
    if (file) {
      setSelectedFiles(prev => ({
        ...prev,
        [fileType]: file
      }));
      // If user selects a new file, make sure we don't mark it for removal
      setFilesToRemove(prev => ({
        ...prev,
        [fileType]: false
      }));
    } else {
      // If file is cleared, mark it for removal if it exists
      if (formData.store[fileType as keyof typeof formData.store]) {
        setFilesToRemove(prev => ({
          ...prev,
          [fileType]: true
        }));
      }
      // Remove from selected files
      setSelectedFiles(prev => {
        const newFiles = { ...prev };
        delete newFiles[fileType as keyof typeof newFiles];
        return newFiles;
      });
    }
  };

  // Handle form submission
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Basic validation
    if (!formData.name.trim()) {
      toast.error('Store owner name is required');
      return;
    }
    
    if (!formData.email.trim()) {
      toast.error('Email is required');
      return;
    }
    
    if (!formData.store.storeName.trim()) {
      toast.error('Store name is required');
      return;
    }

    try {
      setLoading(true);
      
      const formDataToSend = new FormData();
      
      // Append basic fields (exclude password if empty)
      formDataToSend.append('name', formData.name);
      formDataToSend.append('email', formData.email);
      formDataToSend.append('mobile', formData.mobile);
      if (formData.password.trim()) {
        formDataToSend.append('password', formData.password);
      }
      formDataToSend.append('status', formData.status);
      
      // Append nested objects as JSON strings
      formDataToSend.append('store', JSON.stringify(formData.store));
      formDataToSend.append('location', JSON.stringify(formData.location));
      formDataToSend.append('bank', JSON.stringify(formData.bank));
      formDataToSend.append('settings', JSON.stringify(formData.settings));
      
      // Append files
      Object.entries(selectedFiles).forEach(([key, file]) => {
        if (file) {
          formDataToSend.append(key, file);
        }
      });

      // Add flags for files to be removed
      Object.entries(filesToRemove).forEach(([key, shouldRemove]) => {
        if (shouldRemove) {
          formDataToSend.append(`remove${key.charAt(0).toUpperCase() + key.slice(1)}`, 'true');
        }
      });

      await api.put(`/stores/${id}`, formDataToSend, {
        headers: {
          'Content-Type': 'multipart/form-data'
        }
      });
      
      toast.success('Store updated successfully');
      navigate('/admin/store-managers');
    } catch (error: any) {
      console.error('Error updating store:', error);
      const errorMessage = error.response?.data?.msg || error.message || 'Failed to update store';
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  if (initialLoading) {
    return (
      <Layout title="Edit Store Manager" sidebar={<Sidebar items={adminSidebarItems} />}>
        <div className="flex justify-center items-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      </Layout>
    );
  }

  return (
    <Layout title="Edit Store Manager" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex items-center gap-4">
          <button
            onClick={() => navigate('/admin/store-managers')}
            className="flex items-center text-gray-600 hover:text-gray-800"
          >
            <ArrowLeft className="w-5 h-5 mr-2" />
            Back to Store Managers
          </button>
        </div>

        <form onSubmit={handleSubmit} className="space-y-8">
          {/* Seller Information */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <User className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Seller Information</h3>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Owner Name *
                </label>
                <input
                  type="text"
                  value={formData.name}
                  onChange={(e) => setFormData(prev => ({ ...prev, name: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Email *
                </label>
                <input
                  type="email"
                  value={formData.email}
                  onChange={(e) => setFormData(prev => ({ ...prev, email: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Mobile Number *
                </label>
                <input
                  type="tel"
                  value={formData.mobile}
                  onChange={(e) => setFormData(prev => ({ ...prev, mobile: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  pattern="[0-9]{10}"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  New Password (leave empty to keep current)
                </label>
                <input
                  type="password"
                  value={formData.password}
                  onChange={(e) => setFormData(prev => ({ ...prev, password: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  minLength={6}
                />
              </div>
            </div>
          </div>

          {/* Store Information */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <Building className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Store Information</h3>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Store Name *
                </label>
                <input
                  type="text"
                  value={formData.store.storeName}
                  onChange={(e) => handleInputChange('store', 'storeName', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  PAN Number *
                </label>
                <input
                  type="text"
                  value={formData.store.panNumber}
                  onChange={(e) => handleInputChange('store', 'panNumber', e.target.value.toUpperCase())}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  pattern="[A-Z]{5}[0-9]{4}[A-Z]{1}"
                  placeholder="ABCDE1234F"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Commission (%)
                </label>
                <input
                  type="number"
                  value={formData.store.commission}
                  onChange={(e) => handleInputChange('store', 'commission', Number(e.target.value))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  min="0"
                  max="100"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Status
                </label>
                <select
                  value={formData.status}
                  onChange={(e) => setFormData(prev => ({ ...prev, status: e.target.value as any }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                >
                  <option value="Pending">Pending</option>
                  <option value="Active">Active</option>
                  <option value="Inactive">Inactive</option>
                  <option value="Suspended">Suspended</option>
                </select>
              </div>
              
              <div className="md:col-span-2">
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Description
                </label>
                <textarea
                  value={formData.store.description}
                  onChange={(e) => handleInputChange('store', 'description', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  rows={3}
                />
              </div>
            </div>
          </div>

          {/* Location Information */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <MapPin className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Location Information</h3>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  City *
                </label>
                <input
                  type="text"
                  value={formData.location.city}
                  onChange={(e) => handleInputChange('location', 'city', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  State
                </label>
                <input
                  type="text"
                  value={formData.location.state}
                  onChange={(e) => handleInputChange('location', 'state', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
              </div>
              
              <div className="md:col-span-2">
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Street Address
                </label>
                <input
                  type="text"
                  value={formData.location.street}
                  onChange={(e) => handleInputChange('location', 'street', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Latitude *
                </label>
                <input
                  type="number"
                  step="any"
                  value={formData.location.latitude}
                  onChange={(e) => handleInputChange('location', 'latitude', Number(e.target.value))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Longitude *
                </label>
                <input
                  type="number"
                  step="any"
                  value={formData.location.longitude}
                  onChange={(e) => handleInputChange('location', 'longitude', Number(e.target.value))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
            </div>
          </div>

          {/* Bank Information */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <CreditCard className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Bank Information</h3>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Bank Name *
                </label>
                <input
                  type="text"
                  value={formData.bank.bankName}
                  onChange={(e) => handleInputChange('bank', 'bankName', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Account Number *
                </label>
                <input
                  type="text"
                  value={formData.bank.accountNumber}
                  onChange={(e) => handleInputChange('bank', 'accountNumber', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  IFSC Code *
                </label>
                <input
                  type="text"
                  value={formData.bank.ifscCode}
                  onChange={(e) => handleInputChange('bank', 'ifscCode', e.target.value.toUpperCase())}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  pattern="[A-Z]{4}0[A-Z0-9]{6}"
                  placeholder="ABCD0123456"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Account Holder Name *
                </label>
                <input
                  type="text"
                  value={formData.bank.accountHolderName}
                  onChange={(e) => handleInputChange('bank', 'accountHolderName', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
            </div>
          </div>

          {/* File Uploads */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <Upload className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Documents (Upload new files to replace existing ones)</h3>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Store Logo
                </label>
                <input
                  type="file"
                  accept="image/*"
                  onChange={(e) => handleFileChange('logo', e.target.files?.[0] || null)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  National ID Card
                </label>
                <input
                  type="file"
                  accept="image/*,application/pdf"
                  onChange={(e) => handleFileChange('nationalIdCard', e.target.files?.[0] || null)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Address Proof
                </label>
                <input
                  type="file"
                  accept="image/*,application/pdf"
                  onChange={(e) => handleFileChange('addressProof', e.target.files?.[0] || null)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
              </div>
            </div>
          </div>

          {/* Settings */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <Settings className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Settings</h3>
            </div>
            
            <div className="space-y-4">
              <div className="flex items-center">
                <input
                  type="checkbox"
                  id="requireApproval"
                  checked={formData.settings.requireProductApproval}
                  onChange={(e) => handleInputChange('settings', 'requireProductApproval', e.target.checked)}
                  className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                />
                <label htmlFor="requireApproval" className="ml-2 block text-sm text-gray-700">
                  Require product approval before listing
                </label>
              </div>
              
              <div className="flex items-center">
                <input
                  type="checkbox"
                  id="viewCustomers"
                  checked={formData.settings.viewCustomerDetails}
                  onChange={(e) => handleInputChange('settings', 'viewCustomerDetails', e.target.checked)}
                  className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                />
                <label htmlFor="viewCustomers" className="ml-2 block text-sm text-gray-700">
                  Allow viewing customer details
                </label>
              </div>
            </div>
          </div>

          {/* Submit Button */}
          <div className="flex justify-end space-x-4">
            <button
              type="button"
              onClick={() => navigate('/admin/store-managers')}
              className="px-6 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
            >
              Cancel
            </button>
            <button
              type="submit"
              disabled={loading}
              className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50"
            >
              {loading ? 'Updating...' : 'Update Store Manager'}
            </button>
          </div>
        </form>
      </div>
    </Layout>
  );
};

export default EditStoreManagers;
