import React, { useEffect, useState } from 'react';
import { toast } from 'react-hot-toast';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import api from '../../../services/api';
import { Save } from 'lucide-react';

interface SettingsData {
  _id?: string;
  siteName: string;
  contactEmail: string;
  itemsPerPage: number;
  currency: string;
}

const Settings: React.FC = () => {
  const [settings, setSettings] = useState<SettingsData>({
    siteName: '',
    contactEmail: '',
    itemsPerPage: 10,
    currency: 'USD',
  });
  const [loading, setLoading] = useState(true);
  const [isSaving, setIsSaving] = useState(false);

  useEffect(() => {
    const fetchSettings = async () => {
      try {
        setLoading(true);
        const { data } = await api.get('/settings');
        if (data.data) {
          setSettings(data.data);
        }
      } catch (e: any) {
        // It's okay if it fails, means no settings saved yet
        console.info('No settings found, using defaults.');
      } finally {
        setLoading(false);
      }
    };
    fetchSettings();
  }, []);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value, type } = e.target;
    setSettings(prev => ({ 
        ...prev, 
        [name]: type === 'number' ? parseInt(value, 10) : value 
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsSaving(true);
    try {
      const { data } = await api.post('/settings', settings);
      setSettings(data.data);
      toast.success('Settings saved successfully');
    } catch (e: any) {
      toast.error(e.response?.data?.message || 'Failed to save settings');
    } finally {
      setIsSaving(false);
    }
  };

  return (
    <Layout title="Application Settings" sidebar={<Sidebar items={adminSidebarItems} />}>
        {loading ? (
            <div className="text-center py-12">Loading settings...</div>
        ) : (
            <div className="max-w-4xl mx-auto bg-white p-8 rounded-lg shadow">
                <form onSubmit={handleSubmit} className="space-y-6">
                    <h2 className="text-2xl font-bold text-gray-800 mb-6">General Settings</h2>
                    
                    <div>
                        <label htmlFor="siteName" className="block text-sm font-medium text-gray-700">Site Name</label>
                        <input
                            type="text"
                            id="siteName"
                            name="siteName"
                            value={settings.siteName}
                            onChange={handleInputChange}
                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        />
                    </div>

                    <div>
                        <label htmlFor="contactEmail" className="block text-sm font-medium text-gray-700">Contact Email</label>
                        <input
                            type="email"
                            id="contactEmail"
                            name="contactEmail"
                            value={settings.contactEmail}
                            onChange={handleInputChange}
                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        />
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label htmlFor="itemsPerPage" className="block text-sm font-medium text-gray-700">Items Per Page</label>
                            <input
                                type="number"
                                id="itemsPerPage"
                                name="itemsPerPage"
                                value={settings.itemsPerPage}
                                onChange={handleInputChange}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                            />
                        </div>
                        <div>
                            <label htmlFor="currency" className="block text-sm font-medium text-gray-700">Currency</label>
                            <input
                                type="text"
                                id="currency"
                                name="currency"
                                value={settings.currency}
                                onChange={handleInputChange}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                            />
                        </div>
                    </div>

                    <div className="flex justify-end pt-6 border-t mt-6">
                        <button 
                            type="submit" 
                            disabled={isSaving}
                            className="inline-flex items-center px-6 py-3 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 disabled:bg-blue-300"
                        >
                            <Save className="-ml-1 mr-2 h-5 w-5" />
                            {isSaving ? 'Saving...' : 'Save Settings'}
                        </button>
                    </div>
                </form>
            </div>
        )}
    </Layout>
  );
};

export default Settings;
