import React, { useState, useEffect } from 'react';
import Layout from '../../../../components/shared/Layout';
import Sidebar from '../../../../components/shared/Sidebar';
import { adminSidebarItems } from '../../AdminLayout';
import { Search, Plus, Edit, Trash2, Eye, Package } from 'lucide-react';
import { toast } from "react-hot-toast";
import api from "@/services/api";
import { useAuth } from '../../../../context/AuthContext';

interface Unit {
  _id: string;
  name: string;
  short_code?: string;
  remark?: string;
  status: boolean;
  itemCount?: number;
  createdAt: string;
  updatedAt: string;
}

interface UnitFormData {
  name: string;
  short_code: string;
  remark: string;
  status: boolean;
}

const Units: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showViewModal, setShowViewModal] = useState(false);
  const [units, setUnits] = useState<Unit[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedUnit, setSelectedUnit] = useState<Unit | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [formData, setFormData] = useState<UnitFormData>({
    name: '',
    short_code: '',
    remark: '',
    status: true
  });

  // Fetch units from API
  const fetchUnits = async (page = 1, search = '') => {
    try {
      setLoading(true);
      const response = await api.get('/units', {
        params: {
          page,
          limit: 10,
          search
        }
      });
      
      const { data, pagination } = response.data;
      setUnits(data);
      setTotalPages(pagination?.totalPages || 1);
      setCurrentPage(page);
    } catch (error: any) {
      console.error('Error fetching units:', error);
      toast.error(error.response?.data?.msg || 'Failed to fetch units');
    } finally {
      setLoading(false);
    }
  };

  // Create unit
  const createUnit = async (unitData: UnitFormData) => {
    try {
      const response = await api.post('/units', unitData);
      return response.data;
    } catch (error: any) {
      console.error('Error creating unit:', error);
      throw error;
    }
  };

  // Update unit
  const updateUnit = async (id: string, unitData: UnitFormData) => {
    try {
      const response = await api.put(`/units/${id}`, unitData);
      return response.data;
    } catch (error: any) {
      console.error('Error updating unit:', error);
      throw error;
    }
  };

  // Delete unit
  const deleteUnit = async (id: string) => {
    if (!confirm('Are you sure you want to permanently delete this unit? This action cannot be undone.')) {
      return;
    }

    try {
      await api.delete(`/units/${id}`);
      toast.success('Unit permanently deleted successfully');
      fetchUnits(currentPage, searchTerm);
    } catch (error: any) {
      console.error('Error deleting unit:', error);
      const errorMessage = error.response?.data?.msg || error.message || 'Failed to delete unit';
      toast.error(errorMessage);
    }
  };

  const getStatusBadge = (status: boolean) => {
    return status 
      ? 'bg-green-100 text-green-800' 
      : 'bg-red-100 text-red-800';
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  // Reset form
  const resetForm = () => {
    setFormData({
      name: '',
      short_code: '',
      remark: '',
      status: true
    });
    setSelectedUnit(null);
  };

  // Handle form submission
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.name.trim()) {
      toast.error('Unit name is required');
      return;
    }

    try {
      if (selectedUnit) {
        await updateUnit(selectedUnit._id, formData);
        toast.success('Unit updated successfully');
        setShowEditModal(false);
      } else {
        await createUnit(formData);
        toast.success('Unit created successfully');
        setShowAddModal(false);
      }
      resetForm();
      fetchUnits(currentPage, searchTerm);
    } catch (error: any) {
      const errorMessage = error.response?.data?.msg || error.message || 'An error occurred';
      toast.error(errorMessage);
    }
  };

  // Handle edit
  const handleEdit = (unit: Unit) => {
    setSelectedUnit(unit);
    setFormData({
      name: unit.name,
      short_code: unit.short_code || '',
      remark: unit.remark || '',
      status: unit.status
    });
    setShowEditModal(true);
  };

  // Handle search
  const handleSearch = (value: string) => {
    setSearchTerm(value);
    setCurrentPage(1);
    fetchUnits(1, value);
  };

  // Load units on component mount
  useEffect(() => {
    fetchUnits();
  }, []);

  return (
    <Layout title="Units Management" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              type="text"
              placeholder="Search units..."
              className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              value={searchTerm}
              onChange={(e) => handleSearch(e.target.value)}
            />
          </div>
          
          <button 
            onClick={() => setShowAddModal(true)}
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            <Plus className="w-4 h-4 mr-2" />
            Add Unit
          </button>
        </div>

        {/* Loading State */}
        {loading && (
          <div className="flex justify-center items-center py-12">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
          </div>
        )}

        {/* Units Table */}
        {!loading && (
          <div className="bg-white rounded-lg shadow-sm border overflow-hidden">
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Unit
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Short Code
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Remark
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Items
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Actions
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {units.map((unit) => (
                    <tr key={unit._id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          <Package className="w-5 h-5 text-blue-500 mr-3" />
                          <div>
                            <div className="text-sm font-medium text-gray-900">{unit.name}</div>
                            <div className="text-xs text-gray-500">Created {formatDate(unit.createdAt)}</div>
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">
                          {unit.short_code || 'N/A'}
                        </div>
                      </td>
                      <td className="px-6 py-4">
                        <div className="text-sm text-gray-900">
                          {unit.remark || 'No remark available'}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">{unit.itemCount || 0}</div>
                        <div className="text-xs text-gray-500">items</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusBadge(unit.status)}`}>
                          {unit.status ? 'active' : 'inactive'}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                        <div className="flex items-center justify-end space-x-2">
                          <button 
                            onClick={() => {
                              setSelectedUnit(unit);
                              setShowViewModal(true);
                            }}
                            className="text-blue-600 hover:text-blue-900"
                          >
                            <Eye className="w-4 h-4" />
                          </button>
                          <button 
                            onClick={() => handleEdit(unit)}
                            className="text-green-600 hover:text-green-900"
                          >
                            <Edit className="w-4 h-4" />
                          </button>
                          <button 
                            onClick={() => deleteUnit(unit._id)}
                            className="text-red-600 hover:text-red-900"
                          >
                            <Trash2 className="w-4 h-4" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}

        {/* Empty State */}
        {!loading && units.length === 0 && (
          <div className="text-center py-12">
            <Package className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No units found</h3>
            <p className="text-gray-500 mb-4">
              {searchTerm ? 'Try adjusting your search terms' : 'Get started by adding your first unit'}
            </p>
            {!searchTerm && (
              <button 
                onClick={() => setShowAddModal(true)}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
              >
                <Plus className="w-4 h-4 mr-2" />
                Add Unit
              </button>
            )}
          </div>
        )}

        {/* Pagination */}
        {!loading && units.length > 0 && totalPages > 1 && (
          <div className="flex items-center justify-between">
            <div className="text-sm text-gray-700">
              Page {currentPage} of {totalPages}
            </div>
            <div className="flex space-x-2">
              <button
                onClick={() => fetchUnits(currentPage - 1, searchTerm)}
                disabled={currentPage === 1}
                className="px-3 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Previous
              </button>
              <button
                onClick={() => fetchUnits(currentPage + 1, searchTerm)}
                disabled={currentPage === totalPages}
                className="px-3 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Next
              </button>
            </div>
          </div>
        )}

        {/* Add/Edit Unit Modal */}
        {(showAddModal || showEditModal) && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-md max-h-[90vh] overflow-y-auto">
              <h3 className="text-lg font-medium text-gray-900 mb-4">
                {selectedUnit ? 'Edit Unit' : 'Add New Unit'}
              </h3>
              
              <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Unit Name *
                  </label>
                  <input
                    type="text"
                    value={formData.name}
                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder="Enter unit name"
                    required
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Short Code
                  </label>
                  <input
                    type="text"
                    value={formData.short_code}
                    onChange={(e) => setFormData({ ...formData, short_code: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder="Enter short code (e.g., KG, ML)"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Remark
                  </label>
                  <textarea
                    rows={3}
                    value={formData.remark}
                    onChange={(e) => setFormData({ ...formData, remark: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder="Enter unit remark or description"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Status
                  </label>
                  <select 
                    value={formData.status.toString()}
                    onChange={(e) => setFormData({ ...formData, status: e.target.value === 'true' })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  >
                    <option value="true">Active</option>
                    <option value="false">Inactive</option>
                  </select>
                </div>
              </form>
              
              <div className="flex justify-end space-x-3 mt-6">
                <button
                  onClick={() => {
                    setShowAddModal(false);
                    setShowEditModal(false);
                    resetForm();
                  }}
                  className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200"
                >
                  Cancel
                </button>
                <button 
                  onClick={handleSubmit}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                  {selectedUnit ? 'Update Unit' : 'Add Unit'}
                </button>
              </div>
            </div>
          </div>
        )}

        {/* View Unit Modal */}
        {showViewModal && selectedUnit && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-md">
              <h3 className="text-lg font-medium text-gray-900 mb-4">Unit Details</h3>
              
              <div className="space-y-4">
                <div className="flex items-center">
                  <div className="w-16 h-16 bg-gray-100 rounded-lg flex items-center justify-center mr-4">
                    <Package className="w-8 h-8 text-gray-400" />
                  </div>
                  <div>
                    <h4 className="text-xl font-semibold text-gray-900">{selectedUnit.name}</h4>
                    <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusBadge(selectedUnit.status)}`}>
                      {selectedUnit.status ? 'Active' : 'Inactive'}
                    </span>
                  </div>
                </div>

                {selectedUnit.short_code && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Short Code</label>
                    <p className="text-sm text-gray-900">{selectedUnit.short_code}</p>
                  </div>
                )}

                {selectedUnit.remark && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Remark</label>
                    <p className="text-sm text-gray-900">{selectedUnit.remark}</p>
                  </div>
                )}

                <div>
                  <label className="block text-sm font-medium text-gray-700">Items</label>
                  <p className="text-sm text-gray-900">{selectedUnit.itemCount || 0} items</p>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700">Created</label>
                  <p className="text-sm text-gray-900">{formatDate(selectedUnit.createdAt)}</p>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700">Last Updated</label>
                  <p className="text-sm text-gray-900">{formatDate(selectedUnit.updatedAt)}</p>
                </div>
              </div>
              
              <div className="flex justify-end space-x-3 mt-6">
                <button
                  onClick={() => setShowViewModal(false)}
                  className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200"
                >
                  Close
                </button>
                <button 
                  onClick={() => {
                    setShowViewModal(false);
                    handleEdit(selectedUnit);
                  }}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                  Edit Unit
                </button>
              </div>
            </div>
          </div>
        )}
      </div>
    </Layout>
  );
};

export default Units;