import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { toast } from 'react-hot-toast';
import { 
  Plus, 
  Search, 
  Edit, 
  Trash2, 
  Eye, 
  ToggleLeft,
  ToggleRight,
  Percent,
  TrendingUp,
  BarChart3,
  Filter,
  RefreshCw
} from 'lucide-react';
import Layout from "@/components/shared/Layout";
import Sidebar from '@/components/shared/Sidebar';
import { adminSidebarItems } from '../../AdminLayout';

// Types
interface Tax {
  _id: string;
  id: string;
  name: string;
  percentage: number;
  status: boolean;
  itemCount?: number;
  createdAt: string;
  updatedAt: string;
}

interface TaxFormData {
  name: string;
  percentage: number;
  status: boolean;
}

interface PaginationInfo {
  currentPage: number;
  totalPages: number;
  totalItems: number;
  hasNextPage: boolean;
  hasPrevPage: boolean;
}

interface TaxStatistics {
  totalTaxes: number;
  activeTaxes: number;
  inactiveTaxes: number;
  averagePercentage: number;
  minPercentage: number;
  maxPercentage: number;
}

const API_BASE_URL = 'http://localhost:5000/api/v1';

const Taxes: React.FC = () => {
  // State management
  const [taxes, setTaxes] = useState<Tax[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const [showActiveOnly, setShowActiveOnly] = useState(false);
  const [pagination, setPagination] = useState<PaginationInfo>({
    currentPage: 1,
    totalPages: 1,
    totalItems: 0,
    hasNextPage: false,
    hasPrevPage: false
  });
  const [statistics, setStatistics] = useState<TaxStatistics>({
    totalTaxes: 0,
    activeTaxes: 0,
    inactiveTaxes: 0,
    averagePercentage: 0,
    minPercentage: 0,
    maxPercentage: 0
  });

  // Modal states
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showViewModal, setShowViewModal] = useState(false);
  const [selectedTax, setSelectedTax] = useState<Tax | null>(null);

  // Form state
  const [formData, setFormData] = useState<TaxFormData>({
    name: '',
    percentage: 0,
    status: true
  });

  // API functions
  const fetchTaxes = async (page = 1, search = '') => {
    try {
      setLoading(true);
      const response = await axios.get(`${API_BASE_URL}/taxes`, {
        params: { 
          limit: 10, 
          search,
          status: showActiveOnly ? 'active' : 'all'
        },
        headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      
      if (response.data.success) {
        const taxesData = response.data.data || [];
        setTaxes(taxesData);
        setShowActiveOnly(false); // Show all taxes by default
        setPagination(response.data.pagination || {
          currentPage: 1,
          totalPages: 1,
          totalItems: 0,
          hasNextPage: false,
          hasPrevPage: false
        });
        
        if (taxesData.length === 0) {
          console.log('No taxes found in database');
        } else {
          console.log(`Found ${taxesData.length} taxes`);
        }
      }
    } catch (error: any) {
      console.error('Error fetching taxes:', error);
      const errorMessage = error.response?.data?.msg || 'Failed to fetch taxes';
      toast.error(errorMessage);
      setTaxes([]);
    } finally {
      setLoading(false);
    }
  };

  const fetchStatistics = async () => {
    try {
      const response = await axios.get(`${API_BASE_URL}/taxes/analytics/statistics`, {
        headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      
      if (response.data.success) {
        setStatistics(response.data.data);
      }
    } catch (error) {
      console.error('Error fetching statistics:', error);
    }
  };

  const createTax = async (taxData: Omit<TaxFormData, 'status'>) => {
    try {
      const response = await axios.post(`${API_BASE_URL}/taxes`, taxData, {
        headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      
      if (response.data.success) {
        toast.success('Tax created successfully');
        fetchTaxes(currentPage, searchTerm);
        fetchStatistics();
        setShowAddModal(false);
        resetForm();
      }
    } catch (error: any) {
      console.error('Error creating tax:', error);
      const errorMessage = error.response?.data?.msg || 'Failed to create tax';
      toast.error(errorMessage);
    }
  };

  const updateTax = async (id: string, taxData: TaxFormData) => {
    try {
      const response = await axios.put(`${API_BASE_URL}/taxes/${id}`, taxData, {
        headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      
      if (response.data.success) {
        setShowEditModal(false);
        resetForm();
      }
    } catch (error: any) {
      console.error('Error updating tax:', error);
      const errorMessage = error.response?.data?.msg || 'Failed to update tax';
      toast.error(errorMessage);
    }
  };

  const toggleTaxStatus = async (id: string) => {
    try {
      setLoading(true);
      
      // Optimistically update the UI
      setTaxes(prevTaxes => 
        prevTaxes.map(tax => 
          tax._id === id ? { ...tax, status: !tax.status } : tax
        )
      );
      
      const response = await axios.patch(
        `${API_BASE_URL}/taxes/${id}/toggle-status`, 
        {},
        {
          headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
        }
      );
      
      if (response.data.success) {
        // Update statistics
        const updatedTax = response.data.data;
        if (updatedTax) {
          setStatistics(prev => ({
            ...prev,
            activeTaxes: updatedTax.status 
              ? prev.activeTaxes + (prev.inactiveTaxes > 0 ? 1 : 0)
              : Math.max(0, prev.activeTaxes - 1),
            inactiveTaxes: updatedTax.status 
              ? Math.max(0, prev.inactiveTaxes - 1)
              : prev.inactiveTaxes + 1
          }));
        }
        
        toast.success(response.data.msg);
      } else {
        // Revert optimistic update if API call fails
        fetchTaxes(currentPage, searchTerm);
      }
    } catch (error: any) {
      console.error('Error toggling tax status:', error);
      // Revert optimistic update on error
      fetchTaxes(currentPage, searchTerm);
      const errorMessage = error.response?.data?.msg || 'Failed to toggle tax status';
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const deleteTax = async (id: string) => {
    if (!window.confirm('Are you sure you want to delete this tax? This action cannot be undone.')) {
      return;
    }

    try {
      const response = await axios.delete(`${API_BASE_URL}/taxes/${id}`, {
        headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      
      if (response.data.success) {
        toast.success('Tax deleted successfully');
        fetchTaxes(currentPage, searchTerm);
        fetchStatistics();
      }
    } catch (error: any) {
      console.error('Error deleting tax:', error);
      const errorMessage = error.response?.data?.msg || 'Failed to delete tax';
      toast.error(errorMessage);
    }
  };

  // Helper functions
  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  const getStatusBadge = (status: boolean) => {
    return status 
      ? 'bg-green-100 text-green-800' 
      : 'bg-red-100 text-red-800';
  };

  const resetForm = () => {
    setFormData({
      name: '',
      percentage: 0,
      status: true
    });
    setSelectedTax(null);
  };

  const handleAddTax = () => {
    resetForm();
    setShowAddModal(true);
  };

  const handleEditTax = (tax: Tax) => {
    setSelectedTax(tax);
    setFormData({
      name: tax.name,
      percentage: tax.percentage,
      status: tax.status
    });
    setShowEditModal(true);
  };

  const handleViewTax = async (tax: Tax) => {
    try {
      const response = await axios.get(`${API_BASE_URL}/taxes/${tax._id}`);
      if (response.data.success) {
        setSelectedTax(response.data.data);
        setShowViewModal(true);
      }
    } catch (error) {
      console.error('Error fetching tax details:', error);
      toast.error('Failed to fetch tax details');
    }
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.name.trim()) {
      toast.error('Tax name is required');
      return;
    }
    
    if (formData.percentage < 0 || formData.percentage > 100) {
      toast.error('Percentage must be between 0 and 100');
      return;
    }

    if (selectedTax) {
      updateTax(selectedTax._id, formData);
    } else {
      createTax({ name: formData.name, percentage: formData.percentage });
    }
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    setCurrentPage(1);
    fetchTaxes(1, searchTerm);
  };

  const toggleActiveFilter = () => {
    setShowActiveOnly(!showActiveOnly);
    setCurrentPage(1);
    fetchTaxes(1, searchTerm);
  };

  const handlePageChange = (page: number) => {
    setCurrentPage(page);
    fetchTaxes(page, searchTerm);
  };

  // Effects
  useEffect(() => {
    fetchTaxes();
    fetchStatistics();
  }, [showActiveOnly]);

  useEffect(() => {
    const delayedSearch = setTimeout(() => {
      if (searchTerm) {
        setCurrentPage(1);
        fetchTaxes(1, searchTerm);
      } else {
        fetchTaxes(1, '');
      }
    }, 500);

    return () => clearTimeout(delayedSearch);
  }, [searchTerm]);

  // Render functions
  const renderStatistics = () => (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
      <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200">
        <div className="flex items-center justify-between">
          <div>
            <p className="text-sm font-medium text-gray-600">Total Taxes</p>
            <p className="text-2xl font-bold text-gray-900">{statistics.totalTaxes}</p>
          </div>
          <div className="p-3 bg-blue-100 rounded-full">
            <Percent className="h-6 w-6 text-blue-600" />
          </div>
        </div>
      </div>

      <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200">
        <div className="flex items-center justify-between">
          <div>
            <p className="text-sm font-medium text-gray-600">Active Taxes</p>
            <p className="text-2xl font-bold text-green-600">{statistics.activeTaxes}</p>
          </div>
          <div className="p-3 bg-green-100 rounded-full">
            <TrendingUp className="h-6 w-6 text-green-600" />
          </div>
        </div>
      </div>

      <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200">
        <div className="flex items-center justify-between">
          <div>
            <p className="text-sm font-medium text-gray-600">Average Rate</p>
            <p className="text-2xl font-bold text-purple-600">{statistics.averagePercentage.toFixed(1)}%</p>
          </div>
          <div className="p-3 bg-purple-100 rounded-full">
            <BarChart3 className="h-6 w-6 text-purple-600" />
          </div>
        </div>
      </div>

      <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200">
        <div className="flex items-center justify-between">
          <div>
            <p className="text-sm font-medium text-gray-600">Range</p>
            <p className="text-2xl font-bold text-orange-600">{statistics.minPercentage}% - {statistics.maxPercentage}%</p>
          </div>
          <div className="p-3 bg-orange-100 rounded-full">
            <Filter className="h-6 w-6 text-orange-600" />
          </div>
        </div>
      </div>
    </div>
  );

  const renderTaxForm = () => (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <label htmlFor="name" className="block text-sm font-medium text-gray-700 mb-1">
          Tax Name *
        </label>
        <input
          type="text"
          id="name"
          value={formData.name}
          onChange={(e) => setFormData({ ...formData, name: e.target.value })}
          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          placeholder="Enter tax name (e.g., GST 18%)"
          required
        />
      </div>

      <div>
        <label htmlFor="percentage" className="block text-sm font-medium text-gray-700 mb-1">
          Percentage *
        </label>
        <input
          type="number"
          id="percentage"
          min="0"
          max="100"
          step="0.01"
          value={formData.percentage}
          onChange={(e) => setFormData({ ...formData, percentage: parseFloat(e.target.value) || 0 })}
          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          placeholder="Enter percentage (0-100)"
          required
        />
      </div>

      {selectedTax && (
        <div>
          <label className="flex items-center">
            <input
              type="checkbox"
              checked={formData.status}
              onChange={(e) => setFormData({ ...formData, status: e.target.checked })}
              className="rounded border-gray-300 text-blue-600 shadow-sm focus:border-blue-300 focus:ring focus:ring-blue-200 focus:ring-opacity-50"
            />
            <span className="ml-2 text-sm text-gray-700">Active</span>
          </label>
        </div>
      )}

      <div className="flex justify-end space-x-3 pt-4">
        <button
          type="button"
          onClick={() => {
            setShowAddModal(false);
            setShowEditModal(false);
            resetForm();
          }}
          className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
        >
          Cancel
        </button>
        <button
          type="submit"
          className="px-4 py-2 text-sm font-medium text-white bg-blue-600 border border-transparent rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
        >
          {selectedTax ? 'Update Tax' : 'Create Tax'}
        </button>
      </div>
    </form>
  );

  const renderTaxTable = () => (
    <div className="bg-white shadow-sm rounded-lg border border-gray-200">
      <div className="px-6 py-4 border-b border-gray-200">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
          <h3 className="text-lg font-medium text-gray-900">Taxes</h3>
          <div className="mt-2 sm:mt-0 flex items-center space-x-2">
            <button
              onClick={toggleActiveFilter}
              className={`px-3 py-2 text-sm font-medium rounded-md ${
                showActiveOnly 
                  ? 'bg-green-100 text-green-800 hover:bg-green-200' 
                  : 'bg-gray-100 text-gray-800 hover:bg-gray-200'
              }`}
              title={showActiveOnly ? 'Showing active taxes. Click to show all.' : 'Showing all taxes. Click to show active only.'}
            >
              {showActiveOnly ? 'Active Only' : 'All Taxes'}
            </button>
          </div>
          <div className="mt-3 sm:mt-0 flex flex-col sm:flex-row items-start sm:items-center space-y-2 sm:space-y-0 sm:space-x-3">
            <div className="flex items-center space-x-2">
              <button
                type="button"
                onClick={toggleActiveFilter}
                className={`px-3 py-2 text-sm font-medium rounded-md ${
                  showActiveOnly 
                    ? 'bg-green-100 text-green-800 hover:bg-green-200' 
                    : 'bg-gray-100 text-gray-800 hover:bg-gray-200'
                }`}
                title={showActiveOnly ? 'Showing active taxes. Click to show all.' : 'Showing all taxes. Click to show active only.'}
              >
                {showActiveOnly ? 'Active Only' : 'All Taxes'}
              </button>
            </div>
            <form onSubmit={handleSearch} className="flex items-center space-x-2">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                <input
                  type="text"
                  placeholder="Search taxes..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10 pr-4 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
                />
              </div>
              <button
                type="submit"
                className="px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
              >
                Search
              </button>
            </form>
            <button
              onClick={() => {
                setSearchTerm('');
                setCurrentPage(1);
                fetchTaxes(1, '');
              }}
              className="p-2 text-gray-400 hover:text-gray-600"
              title="Refresh"
            >
              <RefreshCw className="h-4 w-4" />
            </button>
            <button
              onClick={handleAddTax}
              className="inline-flex items-center px-4 py-2 text-sm font-medium text-white bg-blue-600 border border-transparent rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
            >
              <Plus className="h-4 w-4 mr-2" />
              Add Tax
            </button>
          </div>
        </div>
      </div>

      {loading ? (
        <div className="flex items-center justify-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      ) : !taxes || taxes.length === 0 ? (
        <div className="text-center py-12">
          <Percent className="mx-auto h-12 w-12 text-gray-400" />
          <h3 className="mt-2 text-sm font-medium text-gray-900">No taxes found</h3>
          <p className="mt-1 text-sm text-gray-500">
            {searchTerm ? 'Try adjusting your search criteria.' : 'Get started by creating a new tax.'}
          </p>
          {!searchTerm && (
            <div className="mt-6">
              <button
                onClick={handleAddTax}
                className="inline-flex items-center px-4 py-2 text-sm font-medium text-white bg-blue-600 border border-transparent rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
              >
                <Plus className="h-4 w-4 mr-2" />
                Add Tax
              </button>
            </div>
          )}
        </div>
      ) : (
        <div className="bg-white rounded-lg shadow-sm border overflow-hidden">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Name
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Percentage
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Status
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Items
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Created
                  </th>
                  <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Actions
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {taxes && taxes.map((tax) => (
                  <tr key={tax._id} className="hover:bg-gray-50">
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex items-center">
                        <Percent className="w-5 h-5 text-blue-500 mr-3" />
                        <div>
                          <div className="text-sm font-medium text-gray-900">{tax.name}</div>
                          <div className="text-xs text-gray-500">Created {formatDate(tax.createdAt)}</div>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="text-sm font-medium text-gray-900">{tax.percentage}%</div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusBadge(tax.status)}`}>
                        {tax.status ? 'active' : 'inactive'}
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="text-sm font-medium text-gray-900">{tax.itemCount || 0}</div>
                      <div className="text-xs text-gray-500">items</div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                      {formatDate(tax.createdAt)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                      <div className="flex items-center justify-end space-x-2">
                        <button
                          onClick={() => handleViewTax(tax)}
                          className="text-blue-600 hover:text-blue-900"
                          title="View"
                        >
                          <Eye className="h-4 w-4" />
                        </button>
                        <button
                          onClick={() => handleEditTax(tax)}
                          className="text-indigo-600 hover:text-indigo-900"
                          title="Edit"
                        >
                          <Edit className="h-4 w-4" />
                        </button>
                        <button
                          onClick={() => !loading && toggleTaxStatus(tax._id)}
                          disabled={loading}
                          className={`relative ${
                            loading 
                              ? 'text-gray-400 cursor-not-allowed' 
                              : tax.status 
                                ? 'text-green-600 hover:text-green-800' 
                                : 'text-gray-400 hover:text-gray-600'
                          } transition-colors duration-200`}
                          title={loading ? 'Updating...' : tax.status ? 'Click to deactivate' : 'Click to activate'}
                        >
                          {loading ? (
                            <RefreshCw className="h-4 w-4 animate-spin" />
                          ) : tax.status ? (
                            <ToggleRight className="h-5 w-5" />
                          ) : (
                            <ToggleLeft className="h-5 w-5" />
                          )}
                        </button>
                        <button
                          onClick={() => deleteTax(tax._id)}
                          className="text-red-600 hover:text-red-900"
                          title="Delete"
                        >
                          <Trash2 className="h-4 w-4" />
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
          
          {/* Pagination */}
          {pagination.totalPages > 1 && (
            <div className="px-6 py-4 border-t border-gray-200">
              <div className="flex items-center justify-between">
                <div className="text-sm text-gray-700">
                  Showing page {pagination.currentPage} of {pagination.totalPages} ({pagination.totalItems} total items)
                </div>
                <div className="flex items-center space-x-2">
                  <button
                    onClick={() => handlePageChange(currentPage - 1)}
                    disabled={!pagination.hasPrevPage}
                    className="px-3 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                  >
                    Previous
                  </button>
                  <span className="px-3 py-2 text-sm text-gray-700">
                    Page {pagination.currentPage} of {pagination.totalPages}
                  </span>
                  <button
                    onClick={() => handlePageChange(currentPage + 1)}
                    disabled={!pagination.hasNextPage}
                    className="px-3 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                  >
                    Next
                  </button>
                </div>
              </div>
            </div>
          )}
        </div>
      )}
    </div>
  );

  const renderModal = (show: boolean, title: string, children: React.ReactNode) => {
    if (!show) return null;

    return (
      <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
          <div className="flex items-center justify-between mb-4">
            <h3 className="text-lg font-medium text-gray-900">{title}</h3>
            <button
              onClick={() => {
                setShowAddModal(false);
                setShowEditModal(false);
                setShowViewModal(false);
                resetForm();
              }}
              className="text-gray-400 hover:text-gray-600"
            >
              <span className="sr-only">Close</span>
              <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </button>
          </div>
          {children}
        </div>
      </div>
    );
  };

  return (
    <Layout title="Taxes Management" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {renderStatistics()}
        {renderTaxTable()}

        {/* Add Tax Modal */}
        {renderModal(showAddModal, 'Add New Tax', renderTaxForm())}

        {/* Edit Tax Modal */}
        {renderModal(showEditModal, 'Edit Tax', renderTaxForm())}

        {/* View Tax Modal */}
        {renderModal(
          showViewModal,
          'Tax Details',
          selectedTax && (
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700">Name</label>
                <p className="mt-1 text-sm text-gray-900">{selectedTax.name}</p>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700">Percentage</label>
                <p className="mt-1 text-sm text-gray-900">{selectedTax.percentage}%</p>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700">Status</label>
                <span className={`mt-1 inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                  selectedTax.status 
                    ? 'bg-green-100 text-green-800' 
                    : 'bg-red-100 text-red-800'
                }`}>
                  {selectedTax.status ? 'Active' : 'Inactive'}
                </span>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700">Items Using This Tax</label>
                <p className="mt-1 text-sm text-gray-900">{selectedTax.itemCount || 0}</p>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700">Created</label>
                <p className="mt-1 text-sm text-gray-900">{new Date(selectedTax.createdAt).toLocaleString()}</p>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700">Last Updated</label>
                <p className="mt-1 text-sm text-gray-900">{new Date(selectedTax.updatedAt).toLocaleString()}</p>
              </div>
            </div>
          )
        )}
      </div>
    </Layout>
  );
};

export default Taxes;