import React, { useState, useEffect } from 'react';
import Layout from '../../../../components/shared/Layout';
import Sidebar from '../../../../components/shared/Sidebar';
import { adminSidebarItems } from '../../AdminLayout';
import { Search, Plus, Edit, Trash2, Eye, Percent, DollarSign, TrendingUp } from 'lucide-react';
import { toast } from "react-hot-toast";
import api from "@/services/api";

interface Discount {
  _id: string;
  name: string;
  type: 'percentage' | 'fixed';
  value: number;
  status: boolean;
  itemCount?: number;
  createdAt: string;
  updatedAt: string;
}

interface DiscountFormData {
  name: string;
  type: 'percentage' | 'fixed';
  value: number;
  status: boolean;
}

interface DiscountStatistics {
  totalDiscounts: number;
  activeDiscounts: number;
  inactiveDiscounts: number;
  averageValue: number;
  maxValue: number;
  minValue: number;
}

const Discounts: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showViewModal, setShowViewModal] = useState(false);
  const [showStatsModal, setShowStatsModal] = useState(false);
  const [discounts, setDiscounts] = useState<Discount[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedDiscount, setSelectedDiscount] = useState<Discount | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [statistics, setStatistics] = useState<DiscountStatistics | null>(null);
  const [formData, setFormData] = useState<DiscountFormData>({
    name: '',
    type: 'percentage',
    value: 0,
    status: true
  });

  // Fetch discounts from API
  const fetchDiscounts = async (page = 1, search = '') => {
    try {
      setLoading(true);
      console.log('Fetching discounts with params:', { page, search });
      
      // If there's a search term, use the search endpoint
      const endpoint = search ? '/discounts/search' : '/discounts';
      const response = await api.get(endpoint, {
        params: search ? { q: search } : { page, limit: 10 }
      });
      
      console.log('API Response:', response.data);
      
      // Handle different response structures
      let discountsData = [];
      if (search) {
        // Search endpoint returns array directly
        discountsData = Array.isArray(response.data) ? response.data : [];
        setTotalPages(1);
        setCurrentPage(1);
      } else {
        // Regular list endpoint with pagination
        const { data, pagination } = response.data;
        discountsData = Array.isArray(data) ? data : [];
        setTotalPages(pagination?.pages || 1);
        setCurrentPage(parseInt(pagination?.page, 10) || 1);
      }
      
      setDiscounts(discountsData);
    } catch (error: any) {
      console.error('Error fetching discounts:', error);
      if (error.response) {
        console.error('Error response data:', error.response.data);
        console.error('Error status:', error.response.status);
      }
      toast.error(error.response?.data?.message || 'Failed to fetch discounts');
      setDiscounts([]);
    } finally {
      setLoading(false);
    }
  };

  // Fetch discount statistics
  const fetchStatistics = async () => {
    try {
      const response = await api.get('/discounts/stats');
      setStatistics(response.data);
    } catch (error: any) {
      console.error('Error fetching statistics:', error);
      toast.error('Failed to fetch statistics');
    }
  };

  // Create discount
  const createDiscount = async (discountData: DiscountFormData) => {
    try {
      const response = await api.post('/discounts', discountData);
      return response.data;
    } catch (error: any) {
      console.error('Error creating discount:', error);
      throw error;
    }
  };

  // Update discount
  const updateDiscount = async (id: string, discountData: DiscountFormData) => {
    try {
      const response = await api.put(`/discounts/${id}`, discountData);
      return response.data;
    } catch (error: any) {
      console.error('Error updating discount:', error);
      throw error;
    }
  };

  // Delete discount
  const deleteDiscount = async (id: string) => {
    if (!confirm('Are you sure you want to permanently delete this discount? This action cannot be undone.')) {
      return;
    }

    try {
      await api.delete(`/discounts/${id}`);
      toast.success('Discount permanently deleted successfully');
      fetchDiscounts(currentPage, searchTerm);
    } catch (error: any) {
      console.error('Error deleting discount:', error);
      const errorMessage = error.response?.data?.msg || error.message || 'Failed to delete discount';
      toast.error(errorMessage);
    }
  };

  const getStatusBadge = (status: boolean) => {
    return status 
      ? 'bg-green-100 text-green-800' 
      : 'bg-red-100 text-red-800';
  };

  const getTypeBadge = (type: string) => {
    return type === 'percentage'
      ? 'bg-blue-100 text-blue-800'
      : 'bg-purple-100 text-purple-800';
  };

  const formatValue = (type: string, value: number) => {
    return type === 'percentage' ? `${value}%` : `₹${value}`;
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  // Reset form
  const resetForm = () => {
    setFormData({
      name: '',
      type: 'percentage',
      value: 0,
      status: true
    });
    setSelectedDiscount(null);
  };

  // Handle form submission
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.name.trim()) {
      toast.error('Discount name is required');
      return;
    }

    if (formData.value < 0) {
      toast.error('Discount value cannot be negative');
      return;
    }

    if (formData.type === 'percentage' && formData.value > 100) {
      toast.error('Percentage value cannot exceed 100%');
      return;
    }

    try {
      if (selectedDiscount) {
        await updateDiscount(selectedDiscount._id, formData);
        toast.success('Discount updated successfully');
        setShowEditModal(false);
      } else {
        await createDiscount(formData);
        toast.success('Discount created successfully');
        setShowAddModal(false);
      }
      resetForm();
      fetchDiscounts(currentPage, searchTerm);
    } catch (error: any) {
      const errorMessage = error.response?.data?.msg || error.message || 'An error occurred';
      toast.error(errorMessage);
    }
  };

  // Handle edit
  const handleEdit = (discount: Discount) => {
    setSelectedDiscount(discount);
    setFormData({
      name: discount.name,
      type: discount.type,
      value: discount.value,
      status: discount.status
    });
    setShowEditModal(true);
  };

  // Handle search
  const handleSearch = (value: string) => {
    setSearchTerm(value);
    setCurrentPage(1);
    fetchDiscounts(1, value);
  };

  // Load discounts on component mount
  useEffect(() => {
    fetchDiscounts();
  }, []);

  return (
    <Layout title="Discounts Management" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              type="text"
              placeholder="Search discounts..."
              className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              value={searchTerm}
              onChange={(e) => handleSearch(e.target.value)}
            />
          </div>
          
          <div className="flex space-x-2">
            <button 
              onClick={() => {
                fetchStatistics();
                setShowStatsModal(true);
              }}
              className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700"
            >
              <TrendingUp className="w-4 h-4 mr-2" />
              Statistics
            </button>
            <button 
              onClick={() => setShowAddModal(true)}
              className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
            >
              <Plus className="w-4 h-4 mr-2" />
              Add Discount
            </button>
          </div>
        </div>

        {/* Loading State */}
        {loading && (
          <div className="flex justify-center items-center py-12">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
          </div>
        )}

        {/* Discounts Table */}
        {!loading && (
          <div className="bg-white rounded-lg shadow-sm border overflow-hidden">
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Discount
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Type
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Value
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Items
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Actions
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {discounts && Array.isArray(discounts) && discounts.map((discount) => (
                    <tr key={discount._id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          {discount.type === 'percentage' ? (
                            <Percent className="w-5 h-5 text-blue-500 mr-3" />
                          ) : (
                            <DollarSign className="w-5 h-5 text-purple-500 mr-3" />
                          )}
                          <div>
                            <div className="text-sm font-medium text-gray-900">{discount.name}</div>
                            <div className="text-xs text-gray-500">Created {formatDate(discount.createdAt)}</div>
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={`px-2 py-1 text-xs font-medium rounded-full ${getTypeBadge(discount.type)}`}>
                          {discount.type}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">
                          {formatValue(discount.type, discount.value)}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">{discount.itemCount || 0}</div>
                        <div className="text-xs text-gray-500">items</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusBadge(discount.status)}`}>
                          {discount.status ? 'active' : 'inactive'}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                        <div className="flex items-center justify-end space-x-2">
                          <button 
                            onClick={() => {
                              setSelectedDiscount(discount);
                              setShowViewModal(true);
                            }}
                            className="text-blue-600 hover:text-blue-900"
                          >
                            <Eye className="w-4 h-4" />
                          </button>
                          <button 
                            onClick={() => handleEdit(discount)}
                            className="text-green-600 hover:text-green-900"
                          >
                            <Edit className="w-4 h-4" />
                          </button>
                          <button 
                            onClick={() => deleteDiscount(discount._id)}
                            className="text-red-600 hover:text-red-900"
                          >
                            <Trash2 className="w-4 h-4" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}

        {/* Empty State */}
        {!loading && (!discounts || discounts.length === 0) && (
          <div className="text-center py-12">
            <Percent className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No discounts found</h3>
            <p className="text-gray-500 mb-4">
              {searchTerm ? 'Try adjusting your search terms' : 'Get started by adding your first discount'}
            </p>
            {!searchTerm && (
              <button 
                onClick={() => setShowAddModal(true)}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
              >
                <Plus className="w-4 h-4 mr-2" />
                Add Discount
              </button>
            )}
          </div>
        )}

        {/* Pagination */}
        {!loading && discounts && discounts.length > 0 && totalPages > 1 && (
          <div className="flex items-center justify-between">
            <div className="text-sm text-gray-700">
              Page {currentPage} of {totalPages}
            </div>
            <div className="flex space-x-2">
              <button
                onClick={() => fetchDiscounts(currentPage - 1, searchTerm)}
                disabled={currentPage === 1}
                className="px-3 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Previous
              </button>
              <select
                value={currentPage}
                onChange={(e) => fetchDiscounts(parseInt(e.target.value), searchTerm)}
                className="px-3 py-2 text-sm border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              >
                {Array.from({ length: totalPages }, (_, i) => i + 1).map(page => (
                  <option key={page} value={page}>
                    {page}
                  </option>
                ))}
              </select>
              <button
                onClick={() => fetchDiscounts(currentPage + 1, searchTerm)}
                disabled={currentPage === totalPages}
                className="px-3 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Next
              </button>
            </div>
          </div>
        )}

        {/* Add/Edit Discount Modal */}
        {(showAddModal || showEditModal) && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-md max-h-[90vh] overflow-y-auto">
              <h3 className="text-lg font-medium text-gray-900 mb-4">
                {selectedDiscount ? 'Edit Discount' : 'Add New Discount'}
              </h3>
              
              <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Discount Name *
                  </label>
                  <input
                    type="text"
                    value={formData.name}
                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder="Enter discount name"
                    required
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Discount Type *
                  </label>
                  <select
                    value={formData.type}
                    onChange={(e) => setFormData({ ...formData, type: e.target.value as 'percentage' | 'fixed' })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    required
                  >
                    <option value="percentage">Percentage</option>
                    <option value="fixed">Fixed Amount</option>
                  </select>
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Value * {formData.type === 'percentage' ? '(%)' : '(₹)'}
                  </label>
                  <input
                    type="number"
                    min="0"
                    max={formData.type === 'percentage' ? 100 : undefined}
                    step={formData.type === 'percentage' ? '0.01' : '1'}
                    value={formData.value}
                    onChange={(e) => setFormData({ ...formData, value: parseFloat(e.target.value) || 0 })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder={formData.type === 'percentage' ? 'Enter percentage (0-100)' : 'Enter amount'}
                    required
                  />
                  {formData.type === 'percentage' && (
                    <p className="text-xs text-gray-500 mt-1">Maximum value is 100%</p>
                  )}
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Status
                  </label>
                  <select 
                    value={formData.status.toString()}
                    onChange={(e) => setFormData({ ...formData, status: e.target.value === 'true' })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  >
                    <option value="true">Active</option>
                    <option value="false">Inactive</option>
                  </select>
                </div>
              </form>
              
              <div className="flex justify-end space-x-3 mt-6">
                <button
                  onClick={() => {
                    setShowAddModal(false);
                    setShowEditModal(false);
                    resetForm();
                  }}
                  className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200"
                >
                  Cancel
                </button>
                <button 
                  onClick={handleSubmit}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                  {selectedDiscount ? 'Update Discount' : 'Add Discount'}
                </button>
              </div>
            </div>
          </div>
        )}

        {/* View Discount Modal */}
        {showViewModal && selectedDiscount && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-md">
              <h3 className="text-lg font-medium text-gray-900 mb-4">Discount Details</h3>
              
              <div className="space-y-4">
                <div className="flex items-center">
                  {selectedDiscount.type === 'percentage' ? (
                    <Percent className="w-8 h-8 text-blue-500 mr-4" />
                  ) : (
                    <DollarSign className="w-8 h-8 text-purple-500 mr-4" />
                  )}
                  <div>
                    <h4 className="text-xl font-semibold text-gray-900">{selectedDiscount.name}</h4>
                    <div className="flex items-center space-x-2 mt-1">
                      <span className={`px-2 py-1 text-xs font-medium rounded-full ${getTypeBadge(selectedDiscount.type)}`}>
                        {selectedDiscount.type}
                      </span>
                      <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusBadge(selectedDiscount.status)}`}>
                        {selectedDiscount.status ? 'Active' : 'Inactive'}
                      </span>
                    </div>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700">Value</label>
                  <p className="text-lg font-semibold text-gray-900">
                    {formatValue(selectedDiscount.type, selectedDiscount.value)}
                  </p>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700">Items Using This Discount</label>
                  <p className="text-sm text-gray-900">{selectedDiscount.itemCount || 0} items</p>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700">Created</label>
                  <p className="text-sm text-gray-900">{formatDate(selectedDiscount.createdAt)}</p>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700">Last Updated</label>
                  <p className="text-sm text-gray-900">{formatDate(selectedDiscount.updatedAt)}</p>
                </div>
              </div>
              
              <div className="flex justify-end space-x-3 mt-6">
                <button
                  onClick={() => setShowViewModal(false)}
                  className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200"
                >
                  Close
                </button>
                <button 
                  onClick={() => {
                    setShowViewModal(false);
                    handleEdit(selectedDiscount);
                  }}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                  Edit Discount
                </button>
              </div>
            </div>
          </div>
        )}

        {/* Statistics Modal */}
        {showStatsModal && statistics && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-2xl">
              <h3 className="text-lg font-medium text-gray-900 mb-4">Discount Statistics</h3>
              
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 mb-6">
                <div className="bg-blue-50 p-4 rounded-lg">
                  <div className="flex items-center">
                    <Percent className="w-8 h-8 text-blue-600 mr-3" />
                    <div>
                      <p className="text-sm font-medium text-blue-600">Total Discounts</p>
                      <p className="text-2xl font-bold text-blue-900">{statistics.totalDiscounts}</p>
                    </div>
                  </div>
                </div>

                <div className="bg-green-50 p-4 rounded-lg">
                  <div className="flex items-center">
                    <TrendingUp className="w-8 h-8 text-green-600 mr-3" />
                    <div>
                      <p className="text-sm font-medium text-green-600">Active Discounts</p>
                      <p className="text-2xl font-bold text-green-900">{statistics.activeDiscounts}</p>
                    </div>
                  </div>
                </div>

                <div className="bg-red-50 p-4 rounded-lg">
                  <div className="flex items-center">
                    <div className="w-8 h-8 bg-red-600 rounded-full flex items-center justify-center mr-3">
                      <span className="text-white text-sm font-bold">!</span>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-red-600">Inactive Discounts</p>
                      <p className="text-2xl font-bold text-red-900">{statistics.inactiveDiscounts}</p>
                    </div>
                  </div>
                </div>

                <div className="bg-purple-50 p-4 rounded-lg">
                  <div className="flex items-center">
                    <DollarSign className="w-8 h-8 text-purple-600 mr-3" />
                    <div>
                      <p className="text-sm font-medium text-purple-600">Average Value</p>
                      <p className="text-2xl font-bold text-purple-900">
                        {statistics.averageValue ? statistics.averageValue.toFixed(2) : '0'}
                      </p>
                    </div>
                  </div>
                </div>

                <div className="bg-orange-50 p-4 rounded-lg">
                  <div className="flex items-center">
                    <TrendingUp className="w-8 h-8 text-orange-600 mr-3" />
                    <div>
                      <p className="text-sm font-medium text-orange-600">Maximum Value</p>
                      <p className="text-2xl font-bold text-orange-900">{statistics.maxValue || 0}</p>
                    </div>
                  </div>
                </div>

                <div className="bg-gray-50 p-4 rounded-lg">
                  <div className="flex items-center">
                    <div className="w-8 h-8 bg-gray-600 rounded-full flex items-center justify-center mr-3">
                      <span className="text-white text-sm font-bold">↓</span>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-600">Minimum Value</p>
                      <p className="text-2xl font-bold text-gray-900">{statistics.minValue || 0}</p>
                    </div>
                  </div>
                </div>
              </div>
              
              <div className="flex justify-end">
                <button
                  onClick={() => setShowStatsModal(false)}
                  className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200"
                >
                  Close
                </button>
              </div>
            </div>
          </div>
        )}
      </div>
    </Layout>
  );
};

export default Discounts;