import React, { useCallback, useEffect, useState } from 'react';
import { Plus, Edit, Trash2, Image as ImageIcon, X, Eye, ArrowUp, ArrowDown } from 'lucide-react';
import { toast } from 'react-hot-toast';

import * as bannerService from '@/services/bannerService';
import { Banner } from '@/types/banner';

// No image placeholder path
const NO_IMAGE = '/images/no-image.png';

// Define banner types as a union of specific string literals
type BannerType = 'main' | 'secondary' | 'promo';

interface BannerCreateDTO {
  title: string;
  subtitle: string;
  image: string;
  link: string;
  isActive: boolean;
  position: number;
  type: BannerType;
}

interface User {
  _id: string;
  email: string;
  token?: string;
}

interface AuthContextType {
  user: User | null;
}

const useAuth = (): AuthContextType => {
  // This is a placeholder - replace with your actual auth context implementation
  return { user: null };
};

// Use standard HTML input and select for now
const Input = React.forwardRef<HTMLInputElement, React.InputHTMLAttributes<HTMLInputElement>>(
  ({ className = '', ...props }, ref) => (
    <input
      ref={ref}
      className={`flex h-10 w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background file:border-0 file:bg-transparent file:text-sm file:font-medium placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50 ${className}`}
      {...props}
    />
  )
);

const Select = React.forwardRef<HTMLSelectElement, React.SelectHTMLAttributes<HTMLSelectElement>>(
  ({ className = '', children, ...props }, ref) => (
    <select
      ref={ref}
      className={`flex h-10 w-full items-center justify-between rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus:outline-none focus:ring-2 focus:ring-ring focus:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50 ${className}`}
      {...props}
    >
      {children}
    </select>
  )
);

const SelectItem = React.forwardRef<HTMLOptionElement, React.OptionHTMLAttributes<HTMLOptionElement>>(
  ({ className = '', children, ...props }, ref) => (
    <option
      ref={ref}
      className={`relative flex w-full cursor-default select-none items-center rounded-sm py-1.5 pl-2 pr-8 text-sm outline-none focus:bg-accent focus:text-accent-foreground data-[disabled]:pointer-events-none data-[disabled]:opacity-50 ${className}`}
      {...props}
    >
      {children}
    </option>
  )
);

// Enhanced Button component to handle variants
const Button = ({ onClick, children, className = '', variant = 'default', disabled = false, ...props }: {
  onClick?: () => void;
  children: React.ReactNode;
  className?: string;
  variant?: 'default' | 'outline' | 'destructive';
  disabled?: boolean;
} & React.ButtonHTMLAttributes<HTMLButtonElement>) => {
  let baseClass = 'px-4 py-2 rounded-md transition-colors ';
  if (variant === 'default') {
    baseClass += 'bg-blue-600 text-white hover:bg-blue-700 ';
  } else if (variant === 'outline') {
    baseClass += 'border border-blue-600 text-blue-600 hover:bg-blue-50 ';
  } else if (variant === 'destructive') {
    baseClass += 'bg-red-600 text-white hover:bg-red-700 ';
  }
  if (disabled) {
    baseClass += 'opacity-50 cursor-not-allowed ';
  }
  return (
    <button
      onClick={onClick}
      className={`${baseClass}${className}`}
      disabled={disabled}
      {...props}
    >
      {children}
    </button>
  );
};

// Simple modal component with size support
interface ModalProps {
  isOpen: boolean;
  onClose: () => void;
  children: React.ReactNode;
  title: string;
  size?: 'sm' | 'default' | '3xl';
}

const Modal: React.FC<ModalProps> = ({ isOpen, onClose, children, title, size = 'default' }) => {
  if (!isOpen) return null;

  let maxWidth = 'max-w-2xl';
  if (size === 'sm') maxWidth = 'max-w-md';
  if (size === '3xl') maxWidth = 'max-w-5xl';

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className={`bg-white rounded-lg shadow-xl w-full ${maxWidth} max-h-[90vh] overflow-y-auto`}>
        <div className="flex justify-between items-center p-4 border-b">
          <h3 className="text-lg font-medium">{title}</h3>
          <button
            onClick={onClose}
            className="text-gray-500 hover:text-gray-700"
            aria-label="Close modal"
          >
            <X className="w-5 h-5" />
          </button>
        </div>
        <div className="p-6">
          {children}
        </div>
      </div>
    </div>
  );
};

const Banners: React.FC = () => {
  // State for banners data and loading
  const [banners, setBanners] = useState<Banner[]>([]);
  const [filteredBanners, setFilteredBanners] = useState<Banner[]>([]);
  const [loading, setLoading] = useState(true);
  const [editingBanner, setEditingBanner] = useState<Banner | null>(null);
  const [selectedBanner, setSelectedBanner] = useState<Banner | null>(null);

  // Modal states
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [isViewModalOpen, setIsViewModalOpen] = useState(false);
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState(false);

  // Form and selection states
  const [searchQuery, setSearchQuery] = useState('');
  const [filter, setFilter] = useState('all');

  // Image handling states
  const [previewImage, setPreviewImage] = useState<string | null>(null);
  const [imageFile, setImageFile] = useState<File | null>(null);

  // Form data state with explicit type
  const [formData, setFormData] = useState<BannerCreateDTO>({
    title: '',
    subtitle: '',
    image: '',
    link: '',
    isActive: true,
    position: 0,
    type: 'main' as const
  });

  // Fetch banners on component mount
  const fetchBanners = async () => {
    try {
      setLoading(true);
      const response = await bannerService.getAllBanners();
      setBanners(response.data);
      setFilteredBanners(response.data);
    } catch (error) {
      console.error('Error fetching banners:', error);
      toast.error('Failed to load banners');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchBanners();
  }, []);

  // Filter banners based on search and filter
  const filterBanners = useCallback(() => {
    let result = [...banners];

    // Apply status filter
    if (filter === 'active') {
      result = result.filter(banner => banner.isActive);
    } else if (filter === 'inactive') {
      result = result.filter(banner => !banner.isActive);
    } else if (filter !== 'all') {
      result = result.filter(banner => banner.type === filter);
    }

    // Apply search
    if (searchQuery) {
      const query = searchQuery.toLowerCase();
      result = result.filter(
        banner =>
          banner.title.toLowerCase().includes(query) ||
          (banner.subtitle && banner.subtitle.toLowerCase().includes(query))
      );
    }

    setFilteredBanners(result);
  }, [banners, filter, searchQuery]);

  // Update filtered banners when dependencies change
  useEffect(() => {
    filterBanners();
  }, [filterBanners]);

  const handleInputChange = (
    e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>
  ) => {
    const { name, value, type } = e.target;
    const checked = (e.target as HTMLInputElement).checked;
    const newValue = type === 'checkbox' ? checked : value;

    setFormData(prev => ({
      ...prev,
      [name]: newValue
    }));
  };

  const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      setImageFile(file);

      // Create preview URL
      const reader = new FileReader();
      reader.onloadend = () => {
        setPreviewImage(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  // Handle form submission
  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();

    try {
      setLoading(true);

      if (editingBanner) {
        // For existing banner, update it first
        const bannerData = { ...formData };
        
        // If there's a new image, upload it first
        if (imageFile) {
          try {
            // For existing banner, we can update the image directly
            const bannerWithImage = await bannerService.uploadBannerImage(editingBanner._id, imageFile);
            bannerData.image = bannerWithImage.image; // Use the new image URL
          } catch (error) {
            console.error('Error uploading image:', error);
            toast.error('Failed to upload new image');
            return;
          }
        }

        // Update the banner with the new data (including the new image URL if uploaded)
        const updatedBanner = await bannerService.updateBanner(
          editingBanner._id,
          bannerData
        );

        setBanners(prev =>
          prev.map(banner =>
            banner._id === editingBanner._id ? updatedBanner.data : banner
          )
        );
        toast.success('Banner updated successfully');
      } else {
        // For new banner, we need to upload the image first if it exists
        let imageUrl = '';
        
        if (imageFile) {
          try {
            // Create a temporary banner with empty image URL
            const tempBanner = await bannerService.createBanner({
              ...formData,
              image: 'temporary-image-placeholder.jpg' // Temporary placeholder
            });
            
            // Upload the image
            const bannerWithImage = await bannerService.uploadBannerImage(tempBanner.data._id, imageFile);
            imageUrl = bannerWithImage.image;
            
            // Update the banner with the actual image URL
            const updatedBanner = await bannerService.updateBanner(
              tempBanner.data._id,
              { ...formData, image: imageUrl }
            );
            
            setBanners(prev => [...prev, updatedBanner.data]);
            toast.success('Banner created successfully');
          } catch (error) {
            console.error('Error creating banner with image:', error);
            toast.error('Failed to create banner with image');
            return;
          }
        } else {
          // If no image is provided, just create the banner with the form data
          const newBanner = await bannerService.createBanner({
            ...formData,
            image: 'default-banner.jpg' // Provide a default image
          });
          setBanners(prev => [...prev, newBanner.data]);
          toast.success('Banner created successfully');
        }
      }

      setIsModalOpen(false);
      resetForm();
    } catch (error) {
      console.error('Error saving banner:', error);
      toast.error('Failed to save banner');
    } finally {
      setLoading(false);
    }
  };

  // Handle banner deletion
  const handleDeleteBanner = async () => {
    if (!editingBanner) return;

    setLoading(true);
    try {
      await bannerService.deleteBanner(editingBanner._id);
      
      // Update the banners list by removing the deleted banner
      setBanners(prev => prev.filter(b => b._id !== editingBanner._id));
      setFilteredBanners(prev => prev.filter(b => b._id !== editingBanner._id));
      
      // Close the modal and reset states
      setIsDeleteModalOpen(false);
      setEditingBanner(null);
      
      toast.success('Banner deleted successfully');
    } catch (error: any) {
      console.error('Error deleting banner:', error);
      const errorMessage = error.response?.data?.message || 'Failed to delete banner';
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  // Reset form to initial state
  const resetForm = () => {
    setFormData({
      title: '',
      subtitle: '',
      image: '',
      link: '',
      isActive: true,
      position: 0,
      type: 'main' as const
    });
    setPreviewImage(null);
    setImageFile(null);
    setEditingBanner(null);
    setSelectedBanner(null);
  };

  const handleEdit = (banner: Banner) => {
    setSelectedBanner(banner);
    setFormData({
      title: banner.title,
      subtitle: banner.subtitle || '',
      image: banner.image,
      link: banner.link || '',
      isActive: banner.isActive,
      position: banner.position,
      type: banner.type
    });
    setPreviewImage(banner.image);
    setIsModalOpen(true);
  };

  // Alias for handleEdit to maintain compatibility with existing code
  const openEditModal = handleEdit;

  const openDeleteModal = (banner: Banner) => {
    setEditingBanner(banner);
    setIsDeleteModalOpen(true);
  };

  const openViewModal = (banner: Banner) => {
    setSelectedBanner(banner);
    setIsViewModalOpen(true);
  };

  const { user } = useAuth();
  
  const moveBanner = async (banner: Banner, direction: 'up' | 'down') => {
    try {
      setLoading(true);
      
      // Get current banner index
      const currentIndex = banners.findIndex(b => b._id === banner._id);
      if (currentIndex === -1) {
        toast.error('Banner not found');
        return;
      }

      // Calculate new position
      const newPosition = direction === 'up' ? currentIndex - 1 : currentIndex + 1;

      // Check if the new position is valid
      if (newPosition < 0 || newPosition >= banners.length) {
        toast.error(`Cannot move banner ${direction === 'up' ? 'up' : 'down'}`);
        return;
      }

      // Create a new array with updated positions
      const updatedBanners = [...banners];
      
      // Swap the banners in the array
      [updatedBanners[currentIndex], updatedBanners[newPosition]] = 
        [updatedBanners[newPosition], updatedBanners[currentIndex]];
      
      // Update positions based on their new array indices
      updatedBanners.forEach((b, index) => {
        b.position = index;
      });

      // Update the backend with new positions
      const updatePromises = updatedBanners.map(b => 
        bannerService.updateBanner(b._id, { position: b.position })
      );
      
      await Promise.all(updatePromises);

      // Update the UI with the new order
      setBanners([...updatedBanners]);
      toast.success(`Banner moved ${direction}`);
    } catch (error) {
      console.error(`Error moving banner ${direction}:`, error);
      toast.error(`Failed to move banner ${direction}`);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="flex justify-between items-center mb-6">
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Banners</h1>
          <p className="mt-1 text-sm text-gray-500">
            Manage your promotional banners and their display settings
          </p>
        </div>
        <Button
          onClick={() => {
            resetForm();
            setIsModalOpen(true);
          }}
        >
          <Plus className="w-4 h-4 mr-2" />
          New Banner
        </Button>
      </div>

      {/* Filters and Search */}
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6 space-y-4 sm:space-y-0 sm:space-x-4">
        <div className="w-full sm:w-auto">
          <label htmlFor="filter" className="block text-sm font-medium text-gray-700 mb-1">
            Filter by Status/Type
          </label>
          <Select
            value={filter}
            onChange={(e) => setFilter(e.target.value)}
          >
            <SelectItem value="all">All Banners</SelectItem>
            <SelectItem value="active">Active</SelectItem>
            <SelectItem value="inactive">Inactive</SelectItem>
            <SelectItem value="main">Main Banners</SelectItem>
            <SelectItem value="secondary">Secondary Banners</SelectItem>
            <SelectItem value="promo">Promo Banners</SelectItem>
          </Select>
        </div>
        <div className="w-full sm:w-auto">
          <label htmlFor="search" className="block text-sm font-medium text-gray-700 mb-1">
            Search Banners
          </label>
          <Input
            type="text"
            id="search"
            placeholder="Search by title or subtitle..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="w-full"
          />
        </div>
      </div>

      {/* Banners Table */}
      <div className="bg-white shadow rounded-lg overflow-hidden">
        {loading && !banners.length ? (
          <div className="p-8 text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto"></div>
            <p className="mt-2 text-gray-600">Loading banners...</p>
          </div>
        ) : filteredBanners.length === 0 ? (
          <div className="p-8 text-center">
            <ImageIcon className="mx-auto h-12 w-12 text-gray-400" />
            <h3 className="mt-2 text-sm font-medium text-gray-900">No banners found</h3>
            <p className="mt-1 text-sm text-gray-500">
              {searchQuery || filter !== 'all'
                ? 'Try adjusting your search or filter to find what you\'re looking for.'
                : 'Get started by creating a new banner.'}
            </p>
            <div className="mt-6">
              <Button
                onClick={() => {
                  resetForm();
                  setIsModalOpen(true);
                }}
              >
                <Plus className="w-4 h-4 mr-2" />
                New Banner
              </Button>
            </div>
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Preview
                  </th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Title
                  </th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Type
                  </th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Status
                  </th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Position
                  </th>
                  <th scope="col" className="relative px-6 py-3">
                    <span className="sr-only">Actions</span>
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {filteredBanners.map((banner) => (
                  <tr key={banner._id} className="hover:bg-gray-50">
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex-shrink-0 h-10 w-16">
                        <img
                          className="h-10 w-16 object-cover rounded-md"
                          src={banner.image ? `${banner.image.startsWith('http') ? '' : '/api/uploads'}${banner.image}` : NO_IMAGE}
                          alt={banner.title}
                          onError={(e) => {
                            const target = e.target as HTMLImageElement;
                            target.src = NO_IMAGE;
                          }}
                        />
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="text-sm font-medium text-gray-900">{banner.title}</div>
                      {banner.subtitle && (
                        <div className="text-sm text-gray-500 truncate max-w-xs">{banner.subtitle}</div>
                      )}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                        banner.type === 'main'
                          ? 'bg-blue-100 text-blue-800'
                          : banner.type === 'secondary'
                          ? 'bg-purple-100 text-purple-800'
                          : 'bg-green-100 text-green-800'
                      }`}>
                        {banner.type.charAt(0).toUpperCase() + banner.type.slice(1)}
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                        banner.isActive ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'
                      }`}>
                        {banner.isActive ? 'Active' : 'Inactive'}
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                      {banner.position + 1}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                      <div className="flex items-center space-x-2 justify-end">
                        <button
                          onClick={() => moveBanner(banner, 'up')}
                          disabled={banners.findIndex(b => b._id === banner._id) === 0}
                          className={`p-1 rounded-full ${
                            banners.findIndex(b => b._id === banner._id) === 0
                              ? 'text-gray-300 cursor-not-allowed'
                              : 'text-gray-400 hover:text-gray-600 hover:bg-gray-100'
                          }`}
                          title="Move up"
                        >
                          <ArrowUp className="h-4 w-4" />
                        </button>
                        <button
                          onClick={() => moveBanner(banner, 'down')}
                          disabled={banners.findIndex(b => b._id === banner._id) === banners.length - 1}
                          className={`p-1 rounded-full ${
                            banners.findIndex(b => b._id === banner._id) === banners.length - 1
                              ? 'text-gray-300 cursor-not-allowed'
                              : 'text-gray-400 hover:text-gray-600 hover:bg-gray-100'
                          }`}
                          title="Move down"
                        >
                          <ArrowDown className="h-4 w-4" />
                        </button>
                        <button
                          onClick={() => openViewModal(banner)}
                          className="text-blue-600 hover:text-blue-900 p-1 rounded-full hover:bg-blue-50"
                          title="View"
                        >
                          <Eye className="h-4 w-4" />
                        </button>
                        <button
                          onClick={() => openEditModal(banner)}
                          className="text-indigo-600 hover:text-indigo-900 p-1 rounded-full hover:bg-indigo-50"
                          title="Edit"
                        >
                          <Edit className="h-4 w-4" />
                        </button>
                        <button
                          onClick={() => openDeleteModal(banner)}
                          className="text-red-600 hover:text-red-900 p-1 rounded-full hover:bg-red-50"
                          title="Delete"
                        >
                          <Trash2 className="h-4 w-4" />
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>

      {/* Add/Edit Banner Modal */}
      <Modal
        isOpen={isModalOpen}
        onClose={() => {
          setIsModalOpen(false);
          resetForm();
        }}
        title={selectedBanner ? 'Edit Banner' : 'Add New Banner'}
      >
        <form onSubmit={handleSubmit}>
          <div className="space-y-4">
            <div>
              <label htmlFor="title" className="block text-sm font-medium text-gray-700">
                Title <span className="text-red-500">*</span>
              </label>
              <Input
                type="text"
                id="title"
                name="title"
                value={formData.title}
                onChange={handleInputChange}
                required
                className="mt-1 block w-full"
              />
            </div>

            <div>
              <label htmlFor="subtitle" className="block text-sm font-medium text-gray-700">
                Subtitle
              </label>
              <Input
                type="text"
                id="subtitle"
                name="subtitle"
                value={formData.subtitle}
                onChange={handleInputChange}
                className="mt-1 block w-full"
              />
            </div>

            <div>
              <label htmlFor="type" className="block text-sm font-medium text-gray-700">
                Banner Type <span className="text-red-500">*</span>
              </label>
              <select
                id="type"
                name="type"
                value={formData.type}
                onChange={handleInputChange}
                required
                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm"
              >
                <option value="main">Main Banner</option>
                <option value="secondary">Secondary Banner</option>
                <option value="promo">Promo Banner</option>
              </select>
            </div>

            <div>
              <label htmlFor="link" className="block text-sm font-medium text-gray-700">
                Link URL (optional)
              </label>
              <Input
                type="url"
                id="link"
                name="link"
                value={formData.link || ''}
                onChange={handleInputChange}
                placeholder="https://example.com"
                className="mt-1 block w-full"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700">
                Banner Image <span className="text-red-500">*</span>
              </label>
              <div className="mt-1 flex items-center">
                <span className="inline-block h-32 w-full rounded-md overflow-hidden bg-gray-100">
                  {previewImage ? (
                    <img
                      src={previewImage}
                      alt="Banner preview"
                      className="h-full w-full object-cover"
                    />
                  ) : selectedBanner?.image ? (
                    <img
                      src={selectedBanner.image.startsWith('http') ? selectedBanner.image : `/api/uploads${selectedBanner.image}`}
                      alt="Current banner"
                      className="h-full w-full object-cover"
                      onError={(e) => {
                        const target = e.target as HTMLImageElement;
                        target.src = NO_IMAGE;
                      }}
                    />
                  ) : (
                    <div className="h-full w-full flex items-center justify-center text-gray-400">
                      <ImageIcon className="h-12 w-12" />
                    </div>
                  )}
                </span>
                <label
                  htmlFor="image-upload"
                  className="ml-3 bg-white py-2 px-3 border border-gray-300 rounded-md shadow-sm text-sm leading-4 font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 cursor-pointer"
                >
                  Change
                </label>
                <input
                  id="image-upload"
                  name="image-upload"
                  type="file"
                  className="sr-only"
                  accept="image/*"
                  onChange={handleImageChange}
                />
              </div>
              <p className="mt-1 text-xs text-gray-500">
                Recommended size: 1200x400px (for main banners) or 600x300px (for secondary/promo banners)
              </p>
            </div>

            <div className="flex items-center">
              <input
                id="isActive"
                name="isActive"
                type="checkbox"
                checked={formData.isActive}
                onChange={handleInputChange}
                className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
              />
              <label htmlFor="isActive" className="ml-2 block text-sm text-gray-700">
                Active
              </label>
            </div>
          </div>

          <div className="mt-6 flex justify-end space-x-3">
            <Button
              type="button"
              variant="outline"
              onClick={() => {
                setIsModalOpen(false);
                resetForm();
              }}
              disabled={loading}
            >
              Cancel
            </Button>
            <Button type="submit" disabled={loading}>
              {loading ? 'Saving...' : 'Save Banner'}
            </Button>
          </div>
        </form>
      </Modal>

      {/* Delete Confirmation Modal */}
      <Modal
        isOpen={isDeleteModalOpen}
        onClose={() => setIsDeleteModalOpen(false)}
        title="Delete Banner"
        size="sm"
      >
        <p className="text-sm text-gray-500">
          Are you sure you want to delete this banner? This action cannot be undone.
        </p>
        <div className="mt-6 flex justify-end space-x-3">
          <Button
            type="button"
            variant="outline"
            onClick={() => setIsDeleteModalOpen(false)}
            disabled={loading}
          >
            Cancel
          </Button>
          <Button
            variant="destructive"
            onClick={handleDeleteBanner}
            disabled={loading}
          >
            {loading ? 'Deleting...' : 'Delete'}
          </Button>
        </div>
      </Modal>

      <Modal
        isOpen={isViewModalOpen}
        onClose={() => setIsViewModalOpen(false)}
        title="Banner Preview"
        size="3xl"
      >
        {selectedBanner && (
          <div className="space-y-6">
            <div className="relative w-full h-64 bg-gray-100 rounded-lg overflow-hidden">
              <img
                src={selectedBanner.image ? `${selectedBanner.image.startsWith('http') ? '' : '/api/uploads'}${selectedBanner.image}` : NO_IMAGE}
                alt={selectedBanner.title}
                className="w-full h-full object-cover"
                onError={(e) => {
                  const target = e.target as HTMLImageElement;
                  target.src = NO_IMAGE;
                }}
              />
            </div>

            <div className="space-y-2">
              <div>
                <h4 className="text-lg font-medium text-gray-900">{selectedBanner.title}</h4>
                {selectedBanner.subtitle && (
                  <p className="text-gray-600">{selectedBanner.subtitle}</p>
                )}
              </div>

              <div className="grid grid-cols-2 gap-4 mt-4">
                <div>
                  <p className="text-sm font-medium text-gray-500">Type</p>
                  <p className="text-sm text-gray-900 capitalize">{selectedBanner.type}</p>
                </div>
                <div>
                  <p className="text-sm font-medium text-gray-500">Status</p>
                  <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                    selectedBanner.isActive
                      ? 'bg-green-100 text-green-800'
                      : 'bg-gray-100 text-gray-800'
                  }`}>
                    {selectedBanner.isActive ? 'Active' : 'Inactive'}
                  </span>
                </div>
                <div>
                  <p className="text-sm font-medium text-gray-500">Position</p>
                  <p className="text-sm text-gray-900">{selectedBanner.position + 1}</p>
                </div>
                {selectedBanner.link && (
                  <div className="col-span-2">
                    <p className="text-sm font-medium text-gray-500">Link</p>
                    <a
                      href={selectedBanner.link}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="text-sm text-blue-600 hover:underline break-all"
                    >
                      {selectedBanner.link}
                    </a>
                  </div>
                )}
              </div>
            </div>

            <div className="pt-4 border-t border-gray-200 mt-6">
              <div className="flex justify-end">
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => setIsViewModalOpen(false)}
                >
                  Close
                </Button>
              </div>
            </div>
          </div>
        )}
      </Modal>
    </div>
  );
};

export default Banners;