import React, { useEffect, useMemo, useState } from 'react';
import { Search, Plus, Edit, Trash2, DollarSign, Calendar, Filter, Download, Eye, CheckCircle, XCircle, Clock } from 'lucide-react';
import { toast } from 'react-hot-toast';
import api from '@/services/api';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';

// Helper function to safely get delivery boy name
const getDeliveryBoyName = (deliveryBoy: any): string => {
  if (!deliveryBoy) return 'N/A';
  if (typeof deliveryBoy === 'string') {
    // This case should ideally be handled by populating data on the backend
    // or fetching details on the frontend. For now, we'll show the ID.
    return deliveryBoy; 
  }
  return deliveryBoy.name || 'Unknown';
};

interface DeliveryBoy {
  _id?: string;
  id?: string;
  name: string;
  phone: string;
}

interface FundsTransfer {
  _id?: string;
  deliveryBoy: DeliveryBoy | string;
  amount: number;
  date: string;
  transactionId: string;
  status: 'pending' | 'completed' | 'failed';
  method: 'bank_transfer' | 'upi' | 'wallet' | 'other';
  notes?: string;
  createdBy: {
    _id: string;
    name: string;
  };
  createdAt: string;
  updatedAt: string;
}

interface FundsTransferForm {
  deliveryBoy: string;
  amount: string;
  method: 'bank_transfer' | 'upi' | 'wallet' | 'other';
  notes: string;
}

const FundsTransfer: React.FC = () => {
  const [fundsTransfers, setFundsTransfers] = useState<FundsTransfer[]>([]);
  const [deliveryBoys, setDeliveryBoys] = useState<DeliveryBoy[] | null>(null);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [deliveryBoyFilter, setDeliveryBoyFilter] = useState('');
  const [startDate, setStartDate] = useState('');
  const [endDate, setEndDate] = useState('');
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState<FundsTransfer | null>(null);
  const [form, setForm] = useState<FundsTransferForm>({
    deliveryBoy: '',
    amount: '',
    method: 'bank_transfer',
    notes: '',
  });

  const hasRows = useMemo(() => fundsTransfers.length > 0, [fundsTransfers]);

  const fetchFundsTransfers = async () => {
    try {
      setLoading(true);
      const params: Record<string, string> = {};
      
      if (searchTerm) params.search = searchTerm;
      if (statusFilter) params.status = statusFilter;
      if (deliveryBoyFilter) params.deliveryBoy = deliveryBoyFilter;
      if (startDate) params.startDate = new Date(startDate).toISOString();
      if (endDate) params.endDate = new Date(endDate).toISOString();

      const res = await api.get('/funds-transfer', { 
        params,
        paramsSerializer: (params) => {
          return Object.entries(params)
            .filter(([_, value]) => value !== '' && value !== null && value !== undefined)
            .map(([key, value]) => `${key}=${encodeURIComponent(value)}`)
            .join('&');
        }
      });
      
      const rows = Array.isArray(res.data?.data) ? res.data.data : 
                  Array.isArray(res.data) ? res.data : [];
      setFundsTransfers(rows);
    } catch (e: any) {
      console.error('Error fetching funds transfers:', e);
      const errorMessage = e.response?.data?.message || e.message || 'Failed to load funds transfers';
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const fetchDeliveryBoys = async () => {
    try {
      const res = await api.get('/delivery-boys?limit=1000'); // Increased limit to ensure we get all delivery boys
      const boys = Array.isArray(res.data?.data) ? res.data.data : 
                  Array.isArray(res.data) ? res.data : [];
      
      // Ensure we have valid delivery boys with required fields
      const validBoys = boys.filter(boy => 
        boy && (boy._id || boy.id) && boy.name
      ).map(boy => ({
        _id: boy._id || boy.id,
        id: boy.id || boy._id,
        name: boy.name || 'Unknown',
        phone: boy.phone || ''
      }));
      
      setDeliveryBoys(validBoys);
    } catch (e: any) {
      console.error('Failed to fetch delivery boys:', e);
      toast.error('Failed to load delivery boys. Please try again.');
      setDeliveryBoys([]);
    }
  };

  useEffect(() => {
    fetchFundsTransfers();
    fetchDeliveryBoys();
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    const amount = parseFloat(form.amount);
    if (isNaN(amount) || amount <= 0) {
      toast.error('Please enter a valid amount greater than 0');
      return;
    }

    setLoading(true);

    try {
      if (editing?._id) {
        // Update logic
        const updateData = {
          amount: amount,
          method: form.method,
          notes: form.notes || undefined,
        };

        console.log('Updating transfer with data:', updateData);
        await api.put(`/funds-transfer/${editing._id}`, updateData);
        toast.success('Funds transfer updated successfully');
      } else {
        // Create logic
        if (!form.deliveryBoy) {
          toast.error('Please select a delivery boy');
          setLoading(false);
          return;
        }

        if (!deliveryBoys || deliveryBoys.length === 0) {
          toast.error('No delivery boys available. Please try again later.');
          setLoading(false);
          return;
        }

        const selectedDeliveryBoy = deliveryBoys.find(
          db => db._id === form.deliveryBoy || db.id === form.deliveryBoy
        );

        if (!selectedDeliveryBoy) {
          toast.error('Selected delivery boy not found');
          setLoading(false);
          return;
        }

        const deliveryBoyId = selectedDeliveryBoy._id || selectedDeliveryBoy.id;
        if (!deliveryBoyId) {
          toast.error('Invalid delivery boy ID');
          setLoading(false);
          return;
        }

        const transferData = {
          deliveryBoy: deliveryBoyId,
          amount: amount,
          method: form.method,
          notes: form.notes || undefined,
        };

        console.log('Creating new transfer with data:', transferData);
        await api.post('/funds-transfer', transferData);
        toast.success('Funds transfer created successfully');
      }

      // Common success logic
      setShowModal(false);
      setEditing(null);
      setForm({ deliveryBoy: '', amount: '', method: 'bank_transfer', notes: '' });
      await fetchFundsTransfers();

    } catch (e: any) {
      console.error('Error saving funds transfer:', e);
      const errorMessage =
        e.response?.data?.message ||
        e.response?.data?.error ||
        'Failed to save funds transfer. Please try again.';
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const handleEdit = async (transfer: FundsTransfer) => {
    console.log('Editing transfer:', transfer);
    setEditing(transfer);
    
    // Ensure we have the latest delivery boys data
    await fetchDeliveryBoys();
    
    // Extract delivery boy ID safely
    let deliveryBoyId = '';
    let deliveryBoyName = 'Unknown';
    let deliveryBoyPhone = '';
    
    if (typeof transfer.deliveryBoy === 'string') {
      deliveryBoyId = transfer.deliveryBoy;
      // Try to find the delivery boy in our existing list
      const existingBoy = deliveryBoys?.find(db => 
        db._id === deliveryBoyId || db.id === deliveryBoyId
      );
      
      if (existingBoy) {
        deliveryBoyName = existingBoy.name || 'Unknown';
        deliveryBoyPhone = existingBoy.phone || '';
      } else {
        // If not found, try to fetch from API
        try {
          const res = await api.get(`/delivery-boys/${deliveryBoyId}`);
          if (res.data) {
            const boy = res.data.data || res.data;
            deliveryBoyName = boy.name || 'Unknown';
            deliveryBoyPhone = boy.phone || '';
            
            // Add to delivery boys list if not already present
            setDeliveryBoys(prev => [
              ...(prev || []),
              {
                _id: boy._id || boy.id,
                id: boy.id || boy._id,
                name: deliveryBoyName,
                phone: deliveryBoyPhone
              }
            ]);
          }
        } catch (e) {
          console.error('Error fetching delivery boy:', e);
        }
      }
    } else if (transfer.deliveryBoy && typeof transfer.deliveryBoy === 'object') {
      // If deliveryBoy is already populated with the object
      const boy = transfer.deliveryBoy;
      deliveryBoyId = (boy as any)._id || (boy as any).id || '';
      deliveryBoyName = boy.name || 'Unknown';
      deliveryBoyPhone = boy.phone || '';
      
      // Add to delivery boys list if not already present
      if (deliveryBoyId && !deliveryBoys?.some(db => 
        db._id === deliveryBoyId || db.id === deliveryBoyId
      )) {
        setDeliveryBoys(prev => [
          ...(prev || []),
          {
            _id: deliveryBoyId,
            id: deliveryBoyId,
            name: deliveryBoyName,
            phone: deliveryBoyPhone
          }
        ]);
      }
    }
    
    // Update form with the transfer data
    setForm({
      deliveryBoy: deliveryBoyId,
      amount: transfer.amount?.toString() || '0',
      method: transfer.method || 'bank_transfer',
      notes: transfer.notes || '',
    });
    
    setShowModal(true);
  };

  const handleStatusUpdate = async (id: string, status: 'pending' | 'completed' | 'failed') => {
    try {
      await api.put(`/funds-transfer/${id}`, { status });
      toast.success(`Status updated to ${status}`);
      fetchFundsTransfers();
    } catch (e: any) {
      toast.error(e.response?.data?.message || 'Failed to update status');
    }
  };

  const handleDelete = async (id: string) => {
    if (!window.confirm('Are you sure you want to delete this funds transfer?')) return;
    
    try {
      await api.delete(`/funds-transfer/${id}`);
      toast.success('Funds transfer deleted successfully');
      fetchFundsTransfers();
    } catch (e: any) {
      toast.error(e.response?.data?.message || 'Failed to delete funds transfer');
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle className="h-4 w-4 text-green-600" />;
      case 'failed':
        return <XCircle className="h-4 w-4 text-red-600" />;
      default:
        return <Clock className="h-4 w-4 text-yellow-600" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'failed':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-yellow-100 text-yellow-800';
    }
  };

  const getMethodLabel = (method: string) => {
    switch (method) {
      case 'bank_transfer':
        return 'Bank Transfer';
      case 'upi':
        return 'UPI';
      case 'wallet':
        return 'Wallet';
      default:
        return 'Other';
    }
  };

  const clearFilters = () => {
    setSearchTerm('');
    setStatusFilter('');
    setDeliveryBoyFilter('');
    setStartDate('');
    setEndDate('');
  };

  return (
    <Layout title="Funds Transfer" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Header with Search and Filters */}
        <div className="bg-white p-6 rounded-lg shadow-sm border">
          <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
            <div className="relative w-full max-w-sm">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                onKeyDown={(e) => e.key === 'Enter' && fetchFundsTransfers()}
                placeholder="Search by transaction ID or delivery boy..."
                className="pl-10 pr-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 w-full"
              />
            </div>
            <button
              onClick={() => {
                console.log('New Transfer button clicked');
                setEditing(null);
                setForm({ deliveryBoy: '', amount: '', method: 'bank_transfer', notes: '' });
                setShowModal(true);
              }}
              className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
            >
              <Plus className="w-4 h-4 mr-2" /> New Transfer
            </button>
          </div>

          {/* Filters */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4 mt-4">
            <select
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value)}
              className="border rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500"
            >
              <option key="all" value="">All Status</option>
              <option key="pending" value="pending">Pending</option>
              <option key="completed" value="completed">Completed</option>
              <option key="failed" value="failed">Failed</option>
            </select>

            <select
              value={deliveryBoyFilter}
              onChange={(e) => setDeliveryBoyFilter(e.target.value)}
              className="border rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500"
            >
              <option key="all-delivery-boys" value="">All Delivery Boys</option>
              {deliveryBoys?.map((boy) => (
                <option key={boy?._id || boy?.id} value={boy?._id || boy?.id}>
                  {boy?.name || 'Unknown'}
                </option>
              )) || null}
            </select>

            <input
              type="date"
              value={startDate}
              onChange={(e) => setStartDate(e.target.value)}
              className="border rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500"
              placeholder="Start Date"
            />

            <input
              type="date"
              value={endDate}
              onChange={(e) => setEndDate(e.target.value)}
              className="border rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500"
              placeholder="End Date"
            />

            <div className="flex gap-2">
              <button
                onClick={fetchFundsTransfers}
                className="flex items-center px-3 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors"
              >
                <Filter className="w-4 h-4 mr-1" /> Filter
              </button>
              <button
                onClick={clearFilters}
                className="px-3 py-2 text-gray-600 hover:text-gray-800 transition-colors"
              >
                Clear
              </button>
            </div>
          </div>
        </div>

        {/* Table */}
        {loading ? (
          <div className="py-12 text-center">Loading...</div>
        ) : !hasRows ? (
          <div className="text-center py-12 bg-white rounded-lg border">
            <DollarSign className="mx-auto h-12 w-12 text-gray-400" />
            <div className="text-gray-600 mt-2">No funds transfers found</div>
          </div>
        ) : (
          <div className="bg-white border rounded-lg overflow-hidden">
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Transaction</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Delivery Boy</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Amount</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Method</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Actions</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {fundsTransfers.map((transfer) => (
                    <tr key={transfer._id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">{transfer.transactionId}</div>
                        {transfer.notes && (
                          <div className="text-xs text-gray-500">{transfer.notes}</div>
                        )}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">
                          {getDeliveryBoyName(transfer.deliveryBoy)}
                        </div>
                        <div className="text-xs text-gray-500">
                          {transfer.deliveryBoy && typeof transfer.deliveryBoy === 'object' 
                            ? transfer.deliveryBoy.phone || 'No phone'
                            : ''}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">₹{transfer.amount.toLocaleString()}</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">{getMethodLabel(transfer.method)}</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center space-x-2">
                          {getStatusIcon(transfer.status)}
                          <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${getStatusColor(transfer.status)}`}>
                            {transfer.status.charAt(0).toUpperCase() + transfer.status.slice(1)}
                          </span>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {new Date(transfer.date).toLocaleDateString()}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                        <div className="flex items-center justify-end space-x-2">
                          {transfer.status === 'pending' && (
                            <>
                              <button
                                onClick={() => handleStatusUpdate(transfer._id!, 'completed')}
                                className="text-green-600 hover:text-green-900 p-1 rounded-full hover:bg-green-50"
                                title="Mark as Completed"
                              >
                                <CheckCircle className="h-4 w-4" />
                              </button>
                              <button
                                onClick={() => handleStatusUpdate(transfer._id!, 'failed')}
                                className="text-red-600 hover:text-red-900 p-1 rounded-full hover:bg-red-50"
                                title="Mark as Failed"
                              >
                                <XCircle className="h-4 w-4" />
                              </button>
                            </>
                          )}
                          <button
                            onClick={() => handleEdit(transfer)}
                            className="text-blue-600 hover:text-blue-900 p-1 rounded-full hover:bg-blue-50"
                            title="Edit"
                          >
                            <Edit className="h-4 w-4" />
                          </button>
                          <button
                            onClick={() => handleDelete(transfer._id!)}
                            className="text-red-600 hover:text-red-900 p-1 rounded-full hover:bg-red-50"
                            title="Delete"
                          >
                            <Trash2 className="h-4 w-4" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}

        {/* Modal */}
        {showModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
            <div className="bg-white rounded-lg p-6 w-full max-w-md">
              <div className="flex justify-between items-center mb-4">
                <h2 className="text-xl font-bold text-gray-800">
                  {editing ? 'Edit Funds Transfer' : 'New Funds Transfer'}
                </h2>
                <button
                  onClick={() => {
                    setShowModal(false);
                    setEditing(null);
                    setForm({ deliveryBoy: '', amount: '', method: 'bank_transfer', notes: '' });
                  }}
                  className="text-gray-500 hover:text-gray-700"
                >
                  ✕
                </button>
              </div>
              
              <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Delivery Boy *</label>
                  <select
                    value={form.deliveryBoy}
                    onChange={(e) => setForm({ ...form, deliveryBoy: e.target.value })}
                    className="w-full border rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500"
                    required
                    disabled={!!editing}
                  >
                    <option key="select-delivery-boy" value="">Select Delivery Boy</option>
                    {Array.isArray(deliveryBoys) && deliveryBoys.map((boy) => (
                      <option key={boy?._id || boy?.id} value={boy?._id || boy?.id}>
                        {boy?.name || 'Unknown'} - {boy?.phone || 'N/A'}
                      </option>
                    ))}
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Amount *</label>
                  <input
                    type="number"
                    step="0.01"
                    min="0"
                    value={form.amount}
                    onChange={(e) => setForm({ ...form, amount: e.target.value })}
                    className="w-full border rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500"
                    placeholder="0.00"
                    required
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Method *</label>
                  <select
                    value={form.method}
                    onChange={(e) => setForm({ ...form, method: e.target.value as any })}
                    className="w-full border rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500"
                    required
                  >
                    <option key="bank_transfer" value="bank_transfer">Bank Transfer</option>
                    <option key="upi" value="upi">UPI</option>
                    <option key="wallet" value="wallet">Wallet</option>
                    <option key="other" value="other">Other</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Notes</label>
                  <textarea
                    value={form.notes}
                    onChange={(e) => setForm({ ...form, notes: e.target.value })}
                    className="w-full border rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500"
                    rows={3}
                    placeholder="Optional notes..."
                  />
                </div>
                
                <div className="flex justify-end space-x-3 pt-4">
                  <button
                    type="button"
                    onClick={() => {
                      setShowModal(false);
                      setEditing(null);
                      setForm({ deliveryBoy: '', amount: '', method: 'bank_transfer', notes: '' });
                    }}
                    className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
                  >
                    Cancel
                  </button>
                  <button
                    type="submit"
                    className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                  >
                    {editing ? 'Update' : 'Create'} Transfer
                  </button>
                </div>
              </form>
            </div>
          </div>
        )}
      </div>
    </Layout>
  );
};

export default FundsTransfer;