import React, { useEffect, useMemo, useState } from 'react';
import { Search, Gift, Plus, Trash2, Edit } from 'lucide-react';
import { toast } from 'react-hot-toast';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import api from '../../../services/api';
import { format } from 'date-fns';

interface Bonus {
  _id: string;
  name: string;
  description?: string;
  bonusType: 'fixed' | 'percentage';
  value: number;
  startDate: string;
  endDate?: string;
  isActive: boolean;
}

const Bonuses: React.FC = () => {
  const [bonuses, setBonuses] = useState<Bonus[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState<Bonus | null>(null);
  const [form, setForm] = useState<Partial<Bonus>>({
    name: '',
    description: '',
    bonusType: 'fixed',
    value: 0,
    startDate: '',
    endDate: '',
    isActive: true,
  });

  const hasRows = useMemo(() => bonuses.length > 0, [bonuses]);

  const fetchBonuses = async () => {
    try {
      setLoading(true);
      const { data } = await api.get('/bonuses', { params: { search: searchTerm } });
      setBonuses(data.data || []);
    } catch (e: any) {
      toast.error(e.response?.data?.message || 'Failed to load bonuses');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchBonuses();
  }, []);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value, type } = e.target as HTMLInputElement;
    const isCheckbox = type === 'checkbox';
    setForm(prev => ({ ...prev!, [name]: isCheckbox ? (e.target as HTMLInputElement).checked : value }));
  };

  const resetForm = () => {
    setEditing(null);
    setForm({
        name: '',
        description: '',
        bonusType: 'fixed',
        value: 0,
        startDate: new Date().toISOString().split('T')[0],
        endDate: '',
        isActive: true,
    });
  };

  const handleAdd = () => {
    resetForm();
    setShowModal(true);
  };

  const handleEdit = (bonus: Bonus) => {
    setEditing(bonus);
    setForm({
      ...bonus,
      startDate: format(new Date(bonus.startDate), 'yyyy-MM-dd'),
      endDate: bonus.endDate ? format(new Date(bonus.endDate), 'yyyy-MM-dd') : '',
    });
    setShowModal(true);
  };

  const handleDelete = async (id: string) => {
    if (window.confirm('Are you sure you want to delete this bonus?')) {
      try {
        await api.delete(`/bonuses/${id}`);
        toast.success('Bonus deleted successfully');
        fetchBonuses();
      } catch (e: any) {
        toast.error(e.response?.data?.message || 'Failed to delete bonus');
      }
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!form.name || !form.startDate) {
        toast.error('Name and Start Date are required.');
        return;
    }

    try {
      const payload = { ...form };
      if (editing) {
        await api.put(`/bonuses/${editing._id}`, payload);
        toast.success('Bonus updated successfully');
      } else {
        await api.post('/bonuses', payload);
        toast.success('Bonus created successfully');
      }
      setShowModal(false);
      fetchBonuses();
    } catch (e: any) {
      toast.error(e.response?.data?.message || 'Failed to save bonus');
    }
  };

  const renderModal = () => (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-lg p-6 w-full max-w-2xl max-h-[90vh] overflow-y-auto">
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-2xl font-bold text-gray-800">{editing ? 'Edit Bonus' : 'Add New Bonus'}</h2>
          <button onClick={() => setShowModal(false)} className="text-gray-500 hover:text-gray-700">✕</button>
        </div>
        <form onSubmit={handleSubmit} className="space-y-4">
            <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Bonus Name *</label>
                <input type="text" name="name" value={form.name || ''} onChange={handleInputChange} className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
            </div>
            <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Description</label>
                <textarea name="description" value={form.description || ''} onChange={handleInputChange} className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" />
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Bonus Type *</label>
                    <select name="bonusType" value={form.bonusType || 'fixed'} onChange={handleInputChange} className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500">
                        <option value="fixed">Fixed</option>
                        <option value="percentage">Percentage</option>
                    </select>
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Value *</label>
                    <input type="number" name="value" value={form.value || 0} onChange={handleInputChange} className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                </div>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Start Date *</label>
                    <input type="date" name="startDate" value={form.startDate || ''} onChange={handleInputChange} className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">End Date</label>
                    <input type="date" name="endDate" value={form.endDate || ''} onChange={handleInputChange} className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" />
                </div>
            </div>
            <div className="flex items-center">
                <input id="isActive" name="isActive" type="checkbox" checked={form.isActive} onChange={handleInputChange} className="focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 rounded" />
                <label htmlFor="isActive" className="ml-3 text-sm font-medium text-gray-700">Active</label>
            </div>
            <div className="flex justify-end space-x-3 pt-6 border-t mt-6">
                <button type="button" onClick={() => setShowModal(false)} className="px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50">Cancel</button>
                <button type="submit" className="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700">{editing ? 'Update Bonus' : 'Create Bonus'}</button>
            </div>
        </form>
      </div>
    </div>
  );

  return (
    <Layout title="Manage Bonuses" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div className="relative w-full max-w-sm">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} onKeyDown={(e) => e.key === 'Enter' && fetchBonuses()} placeholder="Search bonuses..." className="pl-10 pr-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 w-full" />
          </div>
          <button onClick={handleAdd} className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors ml-4">
            <Plus className="w-4 h-4 mr-2" /> Add Bonus
          </button>
        </div>

        {loading ? (
          <div className="py-12 text-center">Loading...</div>
        ) : !hasRows ? (
          <div className="text-center py-12">
            <Gift className="mx-auto h-12 w-12 text-gray-400" />
            <p className="mt-4 text-gray-600">No bonuses found. Get started by adding a new one.</p>
          </div>
        ) : (
          <div className="bg-white border rounded-lg overflow-hidden">
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Name</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Value</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Dates</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Actions</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {bonuses.map((bonus) => (
                    <tr key={bonus._id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">{bonus.name}</div>
                        <div className="text-sm text-gray-500">{bonus.description}</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{bonus.bonusType}</td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{bonus.bonusType === 'percentage' ? `${bonus.value}%` : `$${bonus.value}`}</td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {format(new Date(bonus.startDate), 'MMM dd, yyyy')} - {bonus.endDate ? format(new Date(bonus.endDate), 'MMM dd, yyyy') : 'Ongoing'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${bonus.isActive ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                          {bonus.isActive ? 'Active' : 'Inactive'}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium space-x-2">
                        <button onClick={() => handleEdit(bonus)} className="text-blue-600 hover:text-blue-900 p-1 rounded-full hover:bg-blue-50" title="Edit"><Edit size={16} /></button>
                        <button onClick={() => handleDelete(bonus._id)} className="text-red-600 hover:text-red-900 p-1 rounded-full hover:bg-red-50" title="Delete"><Trash2 size={16} /></button>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}
        {showModal && renderModal()}
      </div>
    </Layout>
  );
};

export default Bonuses;