import { useState, useEffect, useCallback } from 'react';
import { useAuth } from '../context/AuthContext';
import api from '../services/api';
import { toast } from 'react-hot-toast';

const useLocation = () => {
  const { user, isAuthenticated } = useAuth();
  const [location, setLocation] = useState(() => {
    // Load from localStorage for guests or as fallback
    if (typeof window !== 'undefined') {
      const savedLocation = localStorage.getItem('deliveryLocation');
      return savedLocation ? JSON.parse(savedLocation) : null;
    }
    return null;
  });
  const [isLoading, setIsLoading] = useState(false);

  // Load user's saved location on mount if authenticated
  useEffect(() => {
    const loadUserLocation = async () => {
      if (!isAuthenticated) return;
      
      try {
        setIsLoading(true);
        const response = await api.get('/location');
        if (response.data && response.data.data) {
          const { city, state, country } = response.data.data;
          const locationData = {
            displayName: [city, state, country].filter(Boolean).join(', '),
            address: response.data.data
          };
          setLocation(locationData);
          localStorage.setItem('deliveryLocation', JSON.stringify(locationData));
        }
      } catch (error) {
        console.error('Failed to load user location:', error);
      } finally {
        setIsLoading(false);
      }
    };

    loadUserLocation();
  }, [isAuthenticated]);

  // Save location to backend if authenticated, or localStorage if guest
  const saveLocation = useCallback(async (locationData) => {
    const { address, coordinates, displayName } = locationData;
    
    // Prepare location object for storage
    const locationToSave = {
      displayName,
      address: {
        city: address.city || '',
        state: address.state || '',
        country: address.country || 'India',
        coordinates: coordinates || [0, 0],
        formattedAddress: displayName,
        pincode: address.postcode || ''
      }
    };

    try {
      if (isAuthenticated) {
        await api.put('/location', locationToSave.address);
      }
      
      // Save to localStorage for persistence
      localStorage.setItem('deliveryLocation', JSON.stringify(locationToSave));
      setLocation(locationToSave);
      
      toast.success(`Location set to ${displayName}`);
      return true;
    } catch (error) {
      console.error('Failed to save location:', error);
      toast.error('Failed to save location. Using temporary storage only.');
      return false;
    }
  }, [isAuthenticated]);

  // Helper function to format address from OSM data
  const formatAddress = (address) => {
    return {
      city: address.city || address.town || address.village || address.county || '',
      state: address.state || '',
      country: address.country || 'India',
      postcode: address.postcode || '',
      road: address.road || '',
      houseNumber: address.house_number || ''
    };
  };

  // Detect user's current location
  const detectLocation = useCallback(async () => {
    if (!navigator.geolocation) {
      toast.error('Geolocation is not supported by your browser');
      return null;
    }

    try {
      setIsLoading(true);
      
      // Get current position
      const position = await new Promise((resolve, reject) => {
        navigator.geolocation.getCurrentPosition(resolve, reject, {
          enableHighAccuracy: true,
          timeout: 10000,
          maximumAge: 0
        });
      });

      const { latitude, longitude } = position.coords;
      
      // Use OpenStreetMap Nominatim API directly
      const response = await fetch(
        `https://nominatim.openstreetmap.org/reverse?format=json&lat=${latitude}&lon=${longitude}&addressdetails=1&zoom=18`
      );
      
      if (!response.ok) {
        throw new Error('Failed to fetch location data');
      }

      const data = await response.json();
      
      if (data) {
        const { address, display_name } = data;
        const formattedAddress = formatAddress(address);
        
        const locationData = {
          displayName: display_name,
          address: formattedAddress,
          coordinates: [longitude, latitude] // [lng, lat] for GeoJSON
        };
        
        // Save the location
        await saveLocation(locationData);
        return locationData;
      }
      
      throw new Error('Could not determine location');
    } catch (error) {
      console.error('Location detection error:', error);
      const errorMessage = error.message || 'Failed to detect location';
      toast.error(errorMessage);
      return null;
    } finally {
      setIsLoading(false);
    }
  }, [saveLocation]);

  // Clear saved location
  const clearLocation = useCallback(async () => {
    try {
      if (isAuthenticated) {
        await api.delete('/location');
      }
      localStorage.removeItem('deliveryLocation');
      setLocation(null);
      return true;
    } catch (error) {
      console.error('Failed to clear location:', error);
      return false;
    }
  }, [isAuthenticated]);

  return {
    location,
    isLoading,
    detectLocation,
    saveLocation,
    clearLocation
  };
};

export default useLocation;
