import React, { useState, useEffect } from 'react';
import { toast } from 'react-hot-toast';
import ProductCard from './ProductCard';
import { itemService } from '../../services/api';

interface Discount {
  _id: string;
  name: string;
  type: 'percentage' | 'fixed';
  value: number;
}

interface Image {
  url: string;
  is_primary?: boolean;
  position?: number;
  _id?: string;
}

interface Product {
  _id: string;
  id?: string;
  name: string;
  title?: string;
  price: number;
  sales_price?: number;
  discount_id?: Discount;
  images?: Image[];
  image?: string | Image | Image[];
  discount?: string;
  description?: string;
  inStock?: boolean;
  store_id?: string | { _id: string; name: string };
  store_name?: string;
  [key: string]: any; // Add index signature to allow dynamic properties
}

interface ProductSectionProps {
  title: string;
  subtitle?: string;
  productIds?: string[];
  products?: Product[];
}

const ProductSection: React.FC<ProductSectionProps> = ({ 
  title, 
  subtitle,
  productIds = [],
  products: initialProducts = []
}) => {
  const [products, setProducts] = useState<Product[]>(initialProducts);
  const [loading, setLoading] = useState(initialProducts.length === 0);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchProducts = async () => {
      // If products are already provided via props, use them
      if (initialProducts.length > 0) {
        setProducts(initialProducts);
        return;
      }

      // If product IDs are provided, fetch those specific products
      if (productIds.length > 0) {
        try {
          setLoading(true);
          const response = await itemService.getItems(1, productIds.length, undefined, 'relevance');
          
          if (response.success && Array.isArray(response.data)) {
            // Filter products to only include those in the productIds array
            const filteredProducts = response.data.filter((product: Product) => 
              productIds.includes(product._id) || (product.id && productIds.includes(product.id))
            );
            setProducts(filteredProducts);
          } else {
            throw new Error('Invalid response format');
          }
        } catch (err) {
          console.error('Failed to fetch products:', err);
          setError('Failed to load products. Please try again later.');
          toast.error('Failed to load products');
        } finally {
          setLoading(false);
        }
      }
    };

    fetchProducts();
  }, [productIds, initialProducts]);

  if (loading) {
    return (
      <div className="container mx-auto px-4 py-8">
        <h2 className="text-2xl font-bold mb-4">{title}</h2>
        <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 gap-4">
          {[...Array(4)].map((_, i) => (
            <div key={i} className="bg-white rounded-lg shadow-sm overflow-hidden animate-pulse">
              <div className="aspect-w-1 aspect-h-1 bg-gray-200"></div>
              <div className="p-4 space-y-2">
                <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                <div className="h-4 bg-gray-200 rounded w-1/2"></div>
              </div>
            </div>
          ))}
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="container mx-auto px-4 py-8">
        <h2 className="text-2xl font-bold mb-4">{title}</h2>
        <p className="text-red-500">{error}</p>
      </div>
    );
  }

  // Helper function to validate a single product
  const isValidProduct = (product: any): product is Product => {
    try {
      if (!product || typeof product !== 'object') return false;
      
      // Check for required fields with proper type checking
      const hasValidId = Boolean(
        (typeof product._id === 'string' && product._id.trim() !== '') ||
        (typeof product.id === 'string' && product.id.trim() !== '')
      );
      
      const hasRequiredFields = 
        (typeof product.name === 'string' && product.name.trim() !== '') &&
        (typeof product.price === 'number' && !isNaN(product.price));
      
      return hasValidId && hasRequiredFields;
    } catch (e) {
      console.error('Error validating product:', e, product);
      return false;
    }
  };

  // Filter and validate products
  const validProducts = products.filter(isValidProduct);

  // Log detailed information about invalid products for debugging
  if (validProducts.length === 0) {
    if (products.length > 0) {
      console.warn('No valid products to display. Check product data:', {
        totalProducts: products.length,
        invalidProducts: products.filter(p => !isValidProduct(p)),
        validProducts: validProducts.length,
        sampleInvalid: products.find(p => !isValidProduct(p))
      });
    } else {
      console.debug('No products available to display');
    }
    return null;
  }

  const baseUrl = import.meta.env.VITE_API_URL || 'http://localhost:5000';
  
  // Helper function to get the first available image URL from a product
  const getImageUrl = (product: Product): string => {
    if (!product) {
      console.warn('Attempted to get image URL for null/undefined product');
      return '/placeholder-product.png';
    }
    
    // Helper to process a single image (string, Image object, or array of Images)
    const processImage = (img: unknown): string | null => {
      if (!img) return null;
      
      // Handle string URL
      if (typeof img === 'string') {
        const url = img as string;
        return url.startsWith('http') || url.startsWith('/') 
          ? url 
          : `${baseUrl}/${url.replace(/^\/+/, '')}`;
      }
      
      // Handle array of images
      if (Array.isArray(img)) {
        const firstImg = img[0];
        if (typeof firstImg === 'string') {
          const url = firstImg as string;
          return url.startsWith('http') || url.startsWith('/')
            ? url
            : `${baseUrl}/${url.replace(/^\/+/, '')}`;
        }
        // If it's an array of Image objects, get the first one's URL
        return (firstImg as Image)?.url || '';
      }
      
      // Handle single Image object
      return (img as Image)?.url || '';
    };
    
    // Check images array first
    if (Array.isArray(product.images) && product.images.length > 0) {
      const firstImage = product.images[0];
      const url = processImage(firstImage);
      if (url) return url;
    }
    
    // Then check the image property
    if (product.image) {
      const url = processImage(product.image);
      if (url) return url;
    }
    
    // Check for other possible image fields
    const possibleImageFields = ['thumbnail', 'coverImage', 'imageUrl', 'imagePath'];
    for (const field of possibleImageFields) {
      if (product[field as keyof Product]) {
        const url = processImage(product[field as keyof Product] as any);
        if (url) return url;
      }
    }
    
    return '/placeholder-product.png';
  };

  return (
    <section className="py-8 bg-white w-full">
      <div className="w-full px-4 mx-auto max-w-7xl">
        <div className="flex justify-between items-start mb-6">
          <div className="space-y-1">
            <h2 className="text-2xl font-bold text-gray-900">{title}</h2>
            {subtitle && <p className="text-gray-600 text-sm sm:text-base max-w-3xl">{subtitle}</p>}
          </div>
          <a 
            href="/products" 
            className="text-sm sm:text-base font-medium text-primary-600 hover:text-primary-500"
          >
            View all
          </a>
        </div>

        <div className="flex overflow-x-auto pb-4 -mx-2">
          {validProducts.map((product) => {
            // Check if there's a sale price that's lower than the original price
            const hasDiscount = product.sales_price !== undefined && 
                              product.sales_price > 0 && 
                              product.price > product.sales_price;
            
            // Calculate discount percentage if there's a sale
            const discountPercentage = hasDiscount && product.sales_price !== undefined
              ? Math.round(((product.price - product.sales_price) / product.price) * 100)
              : 0;
            
            // Get product title with fallback
            const productTitle = product.name || product.title || 'Unnamed Product';
            
            // Get product image URL with fallback
            const imageUrl = getImageUrl(product);

            return (
              <div 
              key={product._id || product.id || `product-${Math.random().toString(36).substr(2, 9)}`} 
              className="flex-shrink-0 w-1/2 sm:w-1/3 md:w-1/4 lg:w-1/5 xl:w-1/6 px-2"
            >
                <ProductCard
                  id={product._id || product.id || ''}
                  title={productTitle}
                  price={product.sales_price || product.price}
                  originalPrice={product.sales_price && product.price > product.sales_price ? product.price : undefined}
                  image={imageUrl}
                  discount={hasDiscount ? `${discountPercentage}%` : undefined}
                  storeId={
                    product.store_id 
                      ? (typeof product.store_id === 'object' 
                          ? product.store_id?._id 
                          : product.store_id)
                      : undefined
                  }
                  storeName={
                    typeof product.store_id === 'object' 
                      ? product.store_id?.name 
                      : product.store_name
                  }
                />
              </div>
            );
          })}
        </div>
      </div>
    </section>
  );
};

export default ProductSection;
