import { useState, useEffect, useCallback } from 'react';
import { ChevronRight, Plus } from 'lucide-react';
import Header from './Header';
import Footer from './Footer';
import ProductCard from './ProductCard';
import { toast } from 'react-hot-toast';
import useLocation from '../../hooks/useLocation';

interface Location {
  lat: number;
  lng: number;
  address?: string;
  city?: string;
  state?: string;
  country?: string;
  postalCode?: string;
}

interface StoreLocation {
  type: string;
  coordinates: [number, number];
  formattedAddress?: string;
  street?: string;
  city?: string;
  state?: string;
  zipCode?: string;
  country?: string;
}

interface Store {
  _id: string;
  name: string;
  location?: StoreLocation;
  deliveryRadius?: number;
  store?: {
    location: StoreLocation;
    deliveryRadius?: number;
  };
  distanceKm?: number;
  address?: {
    city?: string;
    state?: string;
    country?: string;
    formattedAddress?: string;
  };
}

interface Discount {
  _id: string;
  name: string;
  type: 'percentage' | 'fixed';
  value: number;
}

interface Image {
  url: string;
  // Add other image properties if they exist
}

interface Product {
  _id: string;
  id?: string;
  name: string;
  price: number;
  sales_price?: number;
  discount_id?: Discount;
  images?: Image[];
  description?: string;
  inStock?: boolean;
  store_id?: string | { _id: string; name: string };
  store_name?: string;
  requiresPrescription?: boolean;
  store?: {
    location: {
      coordinates: [number, number];
    };
    deliveryRadius?: number;
  };
}

import { itemService } from '../../services/api';

const filters = [
  'Brands',
  'Health Condition',
  'Diet type',
  'Product form',
  'Benefits',
  'Flavour',
  'Product nature',
  'Age group',
  'Preference',
  'Ingredient',
];

export default function ProductList() {
  const [products, setProducts] = useState<Product[]>([]);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(0);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [sortBy, setSortBy] = useState('relevance');

  // Use the useLocation hook to get store and location
  const { 
    loading: locationLoading, 
    getCurrentLocation,
    nearbyStores
  } = useLocation();

  // State for selected store
  const [selectedStore, setSelectedStore] = useState<any>(() => {
    // Initialize from localStorage if available
    try {
      const savedStore = localStorage.getItem('selectedStore');
      return savedStore ? JSON.parse(savedStore) : null;
    } catch (error) {
      console.error('Error parsing saved store:', error);
      return null;
    }
  });

  // State for user location
  const [userLocation, setUserLocation] = useState<{lat: number; lng: number; address?: string} | null>(() => {
    // Initialize from localStorage if available
    try {
      const savedLocation = localStorage.getItem('userLocation');
      if (savedLocation) {
        const parsed = JSON.parse(savedLocation);
        if (parsed.lat && parsed.lng) {
          console.log('Initializing userLocation from localStorage:', parsed);
          return {
            lat: parsed.lat,
            lng: parsed.lng,
            address: parsed.address
          };
        }
      }
    } catch (error) {
      console.error('Error initializing location from localStorage:', error);
    }
    return null;
  });

  // Update localStorage when selectedStore changes
  useEffect(() => {
    if (selectedStore) {
      try {
        localStorage.setItem('selectedStore', JSON.stringify(selectedStore));
        console.log('Saved selected store to localStorage:', selectedStore);
      } catch (error) {
        console.error('Error saving selected store to localStorage:', error);
      }
    }
  }, [selectedStore]);

  // Update localStorage when userLocation changes
  useEffect(() => {
    if (userLocation) {
      try {
        localStorage.setItem('userLocation', JSON.stringify(userLocation));
      } catch (error) {
        console.error('Error saving user location to localStorage:', error);
      }
    }
  }, [userLocation]);

  // Listen for storage events to update when location changes in other tabs/windows
  useEffect(() => {
    const handleStorageChange = (e: StorageEvent) => {
      if (e.key === 'selectedStore') {
        try {
          if (e.newValue) {
            setSelectedStore(JSON.parse(e.newValue));
            console.log('Updated selected store from storage:', JSON.parse(e.newValue));
          } else {
            setSelectedStore(null);
          }
        } catch (error) {
          console.error('Error parsing updated store:', error);
        }
      } else if (e.key === 'userLocation') {
        try {
          if (e.newValue) {
            const newLocation = JSON.parse(e.newValue);
            setUserLocation(prev => ({
              ...prev,
              ...newLocation
            }));
          } else {
            setUserLocation(null);
          }
        } catch (error) {
          console.error('Error parsing updated location:', error);
        }
      }
    };

    window.addEventListener('storage', handleStorageChange);
    return () => window.removeEventListener('storage', handleStorageChange);
  }, []);

  // Try to detect location if not set
  useEffect(() => {
    if (!userLocation && !locationLoading) {
      console.log('No location found, attempting to detect...');
      getCurrentLocation().catch(err => {
        console.error('Error detecting location:', err);
      });
    }
  }, [userLocation, locationLoading, getCurrentLocation]);
  
  // Log store and location info for debugging
  useEffect(() => {
    console.log('Selected store:', selectedStore);
    console.log('User location state:', userLocation);
    console.log('Nearby stores:', nearbyStores);
    
    if (!selectedStore && nearbyStores?.length > 0) {
      console.log('No store selected, but found nearby stores:', nearbyStores);
    }
  }, [selectedStore, userLocation, nearbyStores]);

  // Helper function to calculate distance between two coordinates (Haversine formula)
  const calculateDistance = (lat1: number, lon1: number, lat2: number, lon2: number): number => {
    const R = 6371e3; // Earth's radius in meters
    const φ1 = (lat1 * Math.PI) / 180;
    const φ2 = (lat2 * Math.PI) / 180;
    const Δφ = ((lat2 - lat1) * Math.PI) / 180;
    const Δλ = ((lon2 - lon1) * Math.PI) / 180;

    const a =
      Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
      Math.cos(φ1) * Math.cos(φ2) * Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

    return R * c; // Distance in meters
  };

  // Fetch complete product data including images and filter by location
  const fetchProductDetails = async (productIds: string[]): Promise<Product[]> => {
    try {
      if (!productIds.length) return [];
      
      // Fetch products with their store information and location data
      const idsParam = productIds.join(',');
      console.log('Fetching products with store data for IDs:', productIds);
      
      // Ensure we're properly populating the store data
      const response = await itemService.getItems(1, productIds.length, `ids=${idsParam}&populate=store`);
      
      // Log the raw response for debugging
      console.log('Raw API response:', response);
      
      // Handle both response formats: response.data.data and response.data
      let products = [];
      if (response?.data && Array.isArray(response.data)) {
        products = response.data;
      } else if (response?.data?.data && Array.isArray(response.data.data)) {
        products = response.data.data;
      } else if (Array.isArray(response)) {
        products = response;
      }
      
      // Log the structure of the first product for debugging
      if (products.length > 0) {
        console.log('First product structure:', JSON.stringify(products[0], null, 2));
      }
      
      console.log('Processed products with store data:', products);
      
      // If we don't have a selected store or user location, show all products
      if (!selectedStore || !userLocation) {
        console.log('No store selected or location not available, showing all products');
        return products;
      }
      
      console.log('Filtering products for store:', selectedStore.name);
      console.log('Selected store object:', selectedStore);
      
      // If the store has a store_id, use that as the store object
      const storeData = selectedStore.store_id || selectedStore.store || selectedStore;
      
      // Try to get store location from different possible locations in the store object
      const storeLocation = 
        storeData.location || 
        (storeData.store_id && storeData.store_id.location) ||
        null;
      
      // If we don't have store location, just show all products
      if (!storeLocation?.coordinates) {
        console.warn('Selected store has no location data, showing all products');
        console.warn('Store object:', storeData);
        return products;
      }
      
      // Extract store coordinates
      let storeLng: number, storeLat: number;
      
      try {
        if (Array.isArray(storeLocation.coordinates)) {
          // MongoDB GeoJSON format: [longitude, latitude]
          [storeLng, storeLat] = storeLocation.coordinates;
          console.log('Using array coordinates:', { storeLng, storeLat });
        } else if (storeLocation.coordinates && 
                  typeof storeLocation.coordinates === 'object' && 
                  'lng' in storeLocation.coordinates && 
                  'lat' in storeLocation.coordinates) {
          // Object format: { lng: number, lat: number }
          storeLng = storeLocation.coordinates.lng;
          storeLat = storeLocation.coordinates.lat;
          console.log('Using object coordinates:', { storeLng, storeLat });
        } else if (storeLocation.coordinates && 
                  storeLocation.coordinates[0] !== undefined && 
                  storeLocation.coordinates[1] !== undefined) {
          // Array-like object: { 0: lng, 1: lat }
          storeLng = storeLocation.coordinates[0];
          storeLat = storeLocation.coordinates[1];
          console.log('Using array-like coordinates:', { storeLng, storeLat });
        } else if (storeLocation.lng !== undefined && storeLocation.lat !== undefined) {
          // Direct lat/lng on the location object
          storeLng = storeLocation.lng;
          storeLat = storeLocation.lat;
          console.log('Using direct lat/lng from location:', { storeLng, storeLat });
        } else {
          console.warn('Unexpected store location format:', storeLocation);
          return products;
        }
      } catch (error) {
        console.error('Error extracting store coordinates:', error);
        return products;
      }
      
      const deliveryRadius = selectedStore.deliveryRadius || 5000; // Default to 5km
      
      console.log(`Filtering products within ${deliveryRadius}m of store location (${storeLat}, ${storeLng})`);
      
      // Filter products based on selected store and location
      return products.filter(product => {
        // Get store data from either store or store_id field
        const productStore = product.store || product.store_id;
        
        // If product has no store data, skip it
        if (!productStore) {
          console.log(`Product '${product.name}' has no store data, skipping`);
          return false;
        }
        
        console.log(`Processing product '${product.name}' with store:`, productStore);
        
        // First, check if the product belongs to the selected store
        const isFromSelectedStore = productStore._id === selectedStore?._id || 
                                 productStore.name === selectedStore?.name;
        
        if (!isFromSelectedStore) {
          console.log(`Product '${product.name}' is not from the selected store '${selectedStore?.name}'`);
          return false;
        }
        
        // Get store location from either store.location or directly from store
        const productStoreLocation = productStore.location || productStore;
        
        // If no coordinates available, use selected store's location
        if (!productStoreLocation?.coordinates) {
          console.log(`Product '${product.name}' has no location data, using selected store location`);
          if (selectedStore?.location?.coordinates) {
            console.log('Using selected store location for distance calculation');
            // Use the selected store's location for this product
            let storeLng, storeLat;
            if (Array.isArray(selectedStore.location.coordinates)) {
              [storeLng, storeLat] = selectedStore.location.coordinates;
            } else {
              storeLng = selectedStore.location.coordinates.lng;
              storeLat = selectedStore.location.coordinates.lat;
            }
            
            const distance = calculateDistance(
              userLocation.lat,
              userLocation.lng,
              storeLat,
              storeLng
            );
            
            const isWithinRadius = distance <= deliveryRadius;
            console.log(`Product '${product.name}' distance: ${distance.toFixed(2)}m (within ${deliveryRadius}m: ${isWithinRadius})`);
            return isWithinRadius;
          }
          // If no store location available, include the product since it's from the selected store
          return true;
        }
        
        // Handle different coordinate formats
        let productLng, productLat;
        try {
          if (Array.isArray(productStoreLocation.coordinates)) {
            // MongoDB GeoJSON format: [longitude, latitude]
            [productLng, productLat] = productStoreLocation.coordinates;
          } else if (productStoreLocation.coordinates.lng !== undefined && 
                    productStoreLocation.coordinates.lat !== undefined) {
            // Object format: { lng: number, lat: number }
            productLng = productStoreLocation.coordinates.lng;
            productLat = productStoreLocation.coordinates.lat;
          } else if (productStoreLocation.coordinates[0] !== undefined && 
                    productStoreLocation.coordinates[1] !== undefined) {
            // Array-like object: { 0: lng, 1: lat }
            productLng = productStoreLocation.coordinates[0];
            productLat = productStoreLocation.coordinates[1];
          } else {
            console.log('Unsupported coordinate format for product:', product.name, productStoreLocation.coordinates);
            return false;
          }
          
          // Calculate distance between store and product
          const distance = calculateDistance(
            storeLat,
            storeLng,
            productLat,
            productLng
          );
          
          const isWithinRange = distance <= deliveryRadius;
          if (!isWithinRange) {
            console.log(`Product '${product.name}' is ${Math.round(distance)}m away (beyond ${deliveryRadius}m radius)`);
          }
          return isWithinRange;
        } catch (error) {
          console.error('Error calculating distance for product:', product._id, error);
          return false; // Exclude product if there's an error
        }
      });
    } catch (error) {
      console.error('Error fetching product details:', error);
      return [];
    }
  };

  // Track if we've already loaded data for the current store
  const [lastFetchedStore, setLastFetchedStore] = useState<string | null>(null);

  // Sort products based on the selected sort option
  const sortProducts = (products: Product[]): Product[] => {
    if (!sortBy) return products;
    
    return [...products].sort((a, b) => {
      const priceA = a.sales_price !== undefined ? a.sales_price : a.price;
      const priceB = b.sales_price !== undefined ? b.sales_price : b.price;
      
      switch (sortBy) {
        case 'price_asc':
          return priceA - priceB;
        case 'price_desc':
          return priceB - priceA;
        case 'newest':
          // Assuming there's a createdAt field, adjust if it's different
          const dateA = new Date(a.createdAt || 0).getTime();
          const dateB = new Date(b.createdAt || 0).getTime();
          return dateB - dateA;
        case 'relevance':
        default:
          // Default sorting (by relevance or as they come from the API)
          return 0;
      }
    });
  };

  // Fetch products with pagination and filtering
  useEffect(() => {
    const fetchProducts = async () => {
      // Skip if we don't have a selected store yet or already fetched for this store
      if (!selectedStore) {
        console.log('No store selected, skipping product fetch');
        return;
      }
      
      // Skip if we've already fetched for this store and page
      if (lastFetchedStore === `${selectedStore._id}-${currentPage}-${sortBy}`) {
        console.log('Already fetched products for store:', selectedStore.name);
        return;
      }

      try {
        setIsLoading(true);
        
        // Build query parameters
        const queryParams: Record<string, string> = {
          page: currentPage.toString(),
          limit: '16',
          status: 'active',
          populate: 'store',
          sort: sortBy || 'relevance'
        };

        // Add store filter
        if (selectedStore?._id) {
          queryParams.storeId = selectedStore._id;
        }

        // Add location if available
        if (userLocation?.lat && userLocation?.lng) {
          queryParams.lat = userLocation.lat.toString();
          queryParams.lng = userLocation.lng.toString();
        }

        console.log('Fetching products with query params:', new URLSearchParams(queryParams).toString());

        const response = await itemService.getItems(currentPage, 16, new URLSearchParams(queryParams).toString());
        
        // Handle response format
        let products: Product[] = [];
        let total = 0;
        
        if (response?.data && Array.isArray(response.data)) {
          products = response.data;
          total = response.pagination?.total || 0;
        } else if (response?.data?.data && Array.isArray(response.data.data)) {
          products = response.data.data;
          total = response.data.pagination?.total || 0;
        } else if (Array.isArray(response)) {
          products = response;
        }

        // Filter products by location if we have a store and location
        if (selectedStore && userLocation) {
          const productIds = products.map(p => p._id);
          let filteredProducts = await fetchProductDetails(productIds);
          // Apply client-side sorting to ensure consistent results
          filteredProducts = sortProducts(filteredProducts);
          setProducts(filteredProducts);
        } else {
          // Still apply client-side sorting for consistency
          const sortedProducts = sortProducts(products);
          setProducts(sortedProducts);
        }
        
        setTotalPages(Math.ceil(total / 16));
        setLastFetchedStore(`${selectedStore._id}-${currentPage}-${sortBy}`);
        
        if (products.length === 0) {
          toast.error(`No products found for ${selectedStore?.name || 'the selected store'}`);
        }
      } catch (err) {
        console.error('Failed to fetch products:', err);
        setError('Failed to load products. Please try again later.');
        setProducts([]);
      } finally {
        setIsLoading(false);
      }
    };

    fetchProducts();
  }, [currentPage, sortBy, selectedStore, userLocation]);

  const handlePageChange = (page: number) => {
    setCurrentPage(page);
  };

  const handleSortChange = (event: React.ChangeEvent<HTMLSelectElement>) => {
    const newSortBy = event.target.value;
    setSortBy(newSortBy);
    // Reset to first page when changing sort order
    setCurrentPage(1);
  };

  if (error) {
    return (
      <div className="min-h-screen bg-white flex flex-col items-center justify-center">
        <p className="text-red-500 text-xl">Failed to load products. Please try again later.</p>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-white flex flex-col">
      <Header />

      <div className="flex-1">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6 sm:py-8">
          <div className="flex items-center space-x-2 text-sm text-gray-600 mb-6">
            <a href="/" className="hover:text-blue-700">Home</a>
            <ChevronRight className="w-4 h-4" />
            <a href="#" className="hover:text-blue-700">Health condition</a>
            <ChevronRight className="w-4 h-4" />
            <span className="text-blue-700 font-medium">Diabetic</span>
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-4 gap-6 lg:gap-8">
            <aside className="lg:col-span-1">
              <div className="lg:sticky lg:top-24">
                <div className="flex items-center justify-between mb-6">
                  <h2 className="text-2xl font-bold text-gray-900">Category</h2>
                  <div className="lg:hidden">
                    <select
                      className="border border-gray-300 rounded-lg px-3 py-2 text-sm"
                      value={sortBy}
                      onChange={handleSortChange}
                    >
                      <option value="relevance">Relevance</option>
                      <option value="price_asc">Price: Low to High</option>
                      <option value="price_desc">Price: High to Low</option>
                      <option value="newest">Newest First</option>
                    </select>
                  </div>
                </div>

                <div className="hidden lg:block">
                  <h3 className="font-bold text-gray-900 mb-4 pb-2 border-b-2 border-gray-900">
                    Filter by
                  </h3>

                  <div className="space-y-1">
                    {filters.map((filter) => (
                      <button
                        key={filter}
                        className="w-full flex items-center justify-between py-3 text-left text-gray-900 hover:bg-gray-50 rounded-lg px-2"
                      >
                        <span className="font-medium">{filter}</span>
                        <Plus className="w-5 h-5 text-gray-600" />
                      </button>
                    ))}
                  </div>
                </div>

                <button className="lg:hidden w-full bg-blue-700 text-white py-3 rounded-lg font-medium mt-4">
                  Show Filters
                </button>
              </div>
            </aside>

            <main className="lg:col-span-3">
              <div className="hidden lg:flex items-center justify-end mb-6">
                <div className="flex items-center space-x-2">
                  <span className="text-sm text-gray-600">Sort by:</span>
                  <select
                    className="border border-gray-300 rounded-lg px-4 py-2 text-sm font-medium text-gray-900"
                    value={sortBy}
                    onChange={handleSortChange}
                  >
                    <option value="relevance">Relevance</option>
                    <option value="price_asc">Price: Low to High</option>
                    <option value="price_desc">Price: High to Low</option>
                    <option value="newest">Newest First</option>
                  </select>
                </div>
              </div>

              {isLoading ? (
                <div className="flex justify-center items-center h-64">
                  <p>Loading products...</p>
                </div>
              ) : (
                <>
                  <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 sm:gap-6 mb-8 px-2 sm:px-4">
                    {products.map((product, index) => {
                      // Calculate discount percentage if sales_price is present
                      const hasDiscount = product.sales_price && product.price > product.sales_price;
                      const discountPercentage = hasDiscount
                        ? Math.round(((product.price - product.sales_price) / product.price) * 100)
                        : 0;

                      // Get the first image URL or use a placeholder
                      const imageUrl = Array.isArray(product.images) && product.images.length > 0
                        ? product.images[0].url
                        : '/placeholder-product.png';

                      // Extract store information
                      const storeId = typeof product.store_id === 'object'
                        ? product.store_id._id
                        : product.store_id;
                      const storeName = typeof product.store_id === 'object'
                        ? product.store_id.name
                        : product.store_name;

                      return (
                        <ProductCard
                          key={product._id || `product-${index}`}
                          id={product._id || ''}
                          title={product.name}
                          price={product.sales_price || product.price}
                          originalPrice={product.sales_price ? product.price : undefined}
                          image={imageUrl}
                          discount={hasDiscount ? `${discountPercentage}%` : undefined}
                          storeId={storeId}
                          storeName={storeName}
                          requiresPrescription={product.requiresPrescription}
                        />
                      );
                    })}
                  </div>

                  <div className="flex items-center justify-center space-x-2">
                    <button
                      onClick={() => handlePageChange(currentPage - 1)}
                      disabled={currentPage === 1}
                      className="w-8 h-8 sm:w-10 sm:h-10 flex items-center justify-center rounded bg-gray-200 text-gray-600 hover:bg-gray-300 disabled:opacity-50"
                    >
                      <ChevronRight className="w-4 h-4 rotate-180" />
                    </button>

                    {[...Array(Math.min(3, totalPages)).keys()].map((i) => {
                      const page = i + 1;
                      return (
                        <button
                          key={page}
                          onClick={() => handlePageChange(page)}
                          className={`w-8 h-8 sm:w-10 sm:h-10 flex items-center justify-center rounded font-medium ${
                            currentPage === page
                              ? 'bg-blue-700 text-white'
                              : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                          }`}
                        >
                          {page}
                        </button>
                      );
                    })}

                    {totalPages > 3 && <span className="text-gray-600">...</span>}

                    {totalPages > 3 && [
                      totalPages - 2,
                      totalPages - 1,
                      totalPages
                    ].map((page) => (
                      <button
                        key={page}
                        onClick={() => handlePageChange(page)}
                        className="w-8 h-8 sm:w-10 sm:h-10 flex items-center justify-center rounded bg-gray-100 text-gray-700 hover:bg-gray-200 font-medium"
                      >
                        {page}
                      </button>
                    ))}

                    <button 
                      onClick={() => handlePageChange(currentPage + 1)}
                      disabled={currentPage === totalPages}
                      className="w-8 h-8 sm:w-10 sm:h-10 flex items-center justify-center rounded bg-gray-200 text-gray-600 hover:bg-gray-300 disabled:opacity-50"
                    >
                      <ChevronRight className="w-4 h-4" />
                    </button>
                  </div>
                </>
              )}
            </main>
          </div>
        </div>
      </div>

      <Footer />
    </div>
  );
}
