const express = require('express');
const router = express.Router();
const { check } = require('express-validator');
const { catchAsync } = require('../middleware/errorHandler');
const { auth } = require('../middleware/auth');
const authController = require('../controllers/authController');
const { protect } = require('../middleware/auth');

// Input validation rules
const registerRules = [
  check('name', 'Name is required').not().isEmpty(),
  check('email', 'Please include a valid email').isEmail(),
  check('password', 'Please enter a password with 6 or more characters').isLength({ min: 6 }),
  check('phone', 'Phone is required').not().isEmpty()
];

const loginRules = [
  check('email', 'Please include a valid email').isEmail(),
  check('password', 'Password is required').exists()
];

// OTP validation rules
const sendOtpRules = [
  check('phone', 'Phone number is required').not().isEmpty(),
  check('phone', 'Please enter a valid 10-digit phone number').isLength({ min: 10, max: 10 })
];

const verifyOtpRules = [
  check('phone', 'Phone number is required').not().isEmpty(),
  check('otp', 'OTP is required').not().isEmpty(),
  check('otp', 'OTP must be 6 digits').isLength({ min: 6, max: 6 })
];

// @route   POST /auth/send-otp
// @desc    Send OTP to phone number
// @access  Public
router.post('/send-otp', sendOtpRules, catchAsync(authController.sendOtp));

// @route   POST /auth/verify-otp
// @desc    Verify OTP and register/login user
// @access  Public
router.post('/verify-otp', verifyOtpRules, catchAsync(authController.verifyOtp));

// @route   POST /auth/register
// @desc    Register a new user (legacy, kept for backward compatibility)
// @access  Public
router.post('/register', registerRules, catchAsync(authController.register));

// @route   POST /auth/login
// @desc    Authenticate user & get token
// @access  Public
router.post('/login', loginRules, catchAsync(authController.login));

// @route   GET /auth/me
// @desc    Get current user's profile
// @access  Private
router.get('/me', auth, catchAsync(authController.getMe));

// Delivery Boy Auth Routes
const deliveryBoyRegisterRules = [
  check('name', 'Name is required').not().isEmpty(),
  check('email', 'Please include a valid email').isEmail(),
  check('password', 'Please enter a password with 6 or more characters').isLength({ min: 6 }),
  check('phone', 'Phone is required').not().isEmpty(),
  check('store', 'Store ID is required').not().isEmpty()
];

const deliveryBoyLoginRules = [
  check('email', 'Please include a valid email').isEmail(),
  check('password', 'Password is required').exists()
];

// @route   POST /auth/delivery-boy/register
// @desc    Register a new delivery boy
// @access  Public
router.post('/delivery-boy/register', deliveryBoyRegisterRules, catchAsync(authController.deliveryBoyRegister));

// @route   POST /auth/delivery-boy/login
// @desc    Authenticate delivery boy & get token
// @access  Public
router.post('/delivery-boy/login', deliveryBoyLoginRules, catchAsync(authController.deliveryBoyLogin));

// @route   GET /auth/delivery-boy/me
// @desc    Get current delivery boy profile
// @access  Private/DeliveryBoy
router.get('/delivery-boy/me', auth, catchAsync(authController.getDeliveryBoyMe));

// @route   POST /auth/forgot-password
// @desc    Forgot password
// @access  Public
router.post('/forgot-password', catchAsync(authController.forgotPassword));

// @route   PUT /auth/reset-password/:token
// @desc    Reset password
// @access  Public
router.put('/reset-password/:token', catchAsync(authController.resetPassword));

// @route   PUT /auth/update-password
// @desc    Update current user's password
// @access  Private
router.put('/update-password', auth, catchAsync(authController.updatePassword));

// @route   POST /auth/logout
// @desc    Logout user / clear cookie
// @access  Private
router.post('/logout', auth, catchAsync(authController.logout));

module.exports = router;
