const mongoose = require('mongoose');

const HomepageSectionSchema = new mongoose.Schema({
  title: {
    type: String,
    required: [true, 'Please provide a section title'],
    trim: true,
    maxlength: [100, 'Section title cannot be more than 100 characters']
  },
  description: {
    type: String,
    trim: true,
    maxlength: [500, 'Section description cannot be more than 500 characters']
  },
  categoryIds: [{
    type: mongoose.Schema.ObjectId,
    ref: 'Category'
  }],
  productType: {
    type: String,
    enum: {
      values: [
        'all_products', 
        'new_added_products', 
        'products_on_sale', 
        'most_selling_products', 
        'custom_products'
      ],
      message: 'Invalid product type'
    },
    default: 'all_products'
  },
  position: {
    type: String,
    enum: {
      values: [
        'top', 
        'below_slider', 
        'below_category', 
        'below_shop_by_seller', 
        'below_shop_by_country', 
        'below_shop_by_brands'
      ],
      message: 'Invalid section position'
    },
    required: [true, 'Please provide a section position']
  },
  backgroundColorLight: {
    type: String,
    default: '#FFFFFF',
    validate: {
      validator: function(v) {
        return /^#([0-9A-F]{3}){1,2}$/i.test(v);
      },
      message: 'Invalid background color format'
    }
  },
  backgroundColorDark: {
    type: String,
    default: '#000000',
    validate: {
      validator: function(v) {
        return /^#([0-9A-F]{3}){1,2}$/i.test(v);
      },
      message: 'Invalid background color format'
    }
  },
  appSectionStyle: {
    type: String,
    enum: {
      values: [
        'grid', 
        'horizontal_scroll', 
        'vertical_scroll', 
        'carousel'
      ],
      message: 'Invalid app section style'
    },
    default: 'grid'
  },
  webSectionStyle: {
    type: String,
    enum: {
      values: [
        'grid', 
        'horizontal_scroll', 
        'vertical_scroll', 
        'carousel'
      ],
      message: 'Invalid web section style'
    },
    default: 'grid'
  },
  products: [{
    type: mongoose.Schema.ObjectId,
    ref: 'Item'
  }],
  isActive: {
    type: Boolean,
    default: true
  },
  sortOrder: {
    type: Number,
    default: 0
  },
  createdAt: {
    type: Date,
    default: Date.now
  },
  updatedAt: {
    type: Date,
    default: Date.now
  }
}, {
  toJSON: { virtuals: true },
  toObject: { virtuals: true }
});

// Populate references when querying
HomepageSectionSchema.pre(/^find/, function(next) {
  this.populate([
    {
      path: 'products',
      select: 'name price image description discount originalPrice'
    },
    {
      path: 'categoryIds',
      select: 'name'
    }
  ]);
  next();
});

// Ensure unique sort order
HomepageSectionSchema.pre('save', async function(next) {
  if (!this.sortOrder) {
    const lastSection = await this.constructor.findOne({}, {}, { sort: { sortOrder: -1 } });
    this.sortOrder = lastSection ? lastSection.sortOrder + 1 : 1;
  }
  next();
});

module.exports = mongoose.model('HomepageSection', HomepageSectionSchema);

